<?php

/**
 * This file is part of Dev5 Integração ERP Bling API V3 for OpenCart
 *
 * @author    Dev5™ <developer@dev5.com.br>
 * @copyright Dev5™ 2025. Todos os direitos reservados.
 * @license   Distribuição/reprodução não autorizada estará sujeita às penas das leis 9609/98 e 9610/98
 * @link      https://dev5.com.br/integracao-erp-bling-api-v3
 * @version   4.0.6
 */

namespace Dev5\Library\Bling;

use Dev5\Client\Http;
use Exception;
use Throwable;

use function Dev5\Opencart\ocapp;
use function Dev5\Opencart\ocver;

/** @Dev5\Orm\Entity(['name' => DB_PREFIX . 'd5bling3_product', 'auto_migrate' => false]) */
final class ProductEntity extends BaseEntity
{
    /**
     * @var int
     * @Dev5\Orm\Field(['type' => 'int', 'primary' => true])
     */
    public $product_id;

    /**
     * @var string
     * @Dev5\Orm\Field(['type' => 'varchar', 'primary' => true, 'length' => 200, 'default' => ''])
     */
    public $options;

    /**
     * @var string|null
     * @Dev5\Orm\Field(['type' => 'varchar', 'length' => 64, 'index' => true, 'nullable' => true])
     */
    public $bling_id;

    /**
     * @var string|null
     * @Dev5\Orm\Field(['type' => 'varchar', 'length' => 64, 'index' => true, 'nullable' => true])
     */
    public $bling_master_id;

    /**
     * @var string|null
     * @Dev5\Orm\Field(['type' => 'varchar', 'length' => 100, 'nullable' => true])
     */
    public $codigo;

    /**
     * @var string|null
     * @Dev5\Orm\Field(['type' => 'varchar', 'length' => 100, 'nullable' => true])
     */
    public $tipo;

    /**
     * @var \DateTime|null
     * @Dev5\Orm\Field(['type' => 'date', 'nullable' => true])
     */
    public $data_validade;

    /**
     * @var string|null
     * @Dev5\Orm\Field(['type' => 'varchar', 'length' => 15, 'nullable' => true])
     */
    public $unidade;

    /**
     * @var string|null
     * @Dev5\Orm\Field(['type' => 'varchar', 'nullable' => true])
     */
    public $deposito;

    /**
     * @var float|null
     * @Dev5\Orm\Field(['type' => 'decimal', 'nullable' => true, 'length' => 10, 'scale' => 4])
     */
    public $peso_liquido;

    /**
     * @var int|null
     * @Dev5\Orm\Field(['type' => 'int', 'nullable' => true])
     */
    public $volumes;

    /**
     * @var int|null
     * @Dev5\Orm\Field(['type' => 'int', 'nullable' => true])
     */
    public $itens_por_caixa;

    /**
     * @var string|null
     * @Dev5\Orm\Field(['type' => 'varchar', 'length' => 100, 'nullable' => true])
     */
    public $gtin;

    /**
     * @var string|null
     * @Dev5\Orm\Field(['type' => 'varchar', 'length' => 100, 'nullable' => true])
     */
    public $gtin_embalagem;

    /**
     * @var string|null
     * @Dev5\Orm\Field(['type' => 'varchar', 'length' => 1, 'nullable' => true])
     */
    public $tipo_producao;

    /**
     * @var int|null
     * @Dev5\Orm\Field(['type' => 'tinyint', 'nullable' => true])
     */
    public $condicao;

    /**
     * @var bool|null
     * @Dev5\Orm\Field(['type' => 'bool', 'nullable' => true])
     */
    public $frete_gratis;

    /**
     * @var string|null
     * @Dev5\Orm\Field(['type' => 'text', 'nullable' => true])
     */
    public $observacoes;

    /**
     * @var string|null
     * @Dev5\Orm\Field(['type' => 'text', 'nullable' => true])
     */
    public $descricao_embalagem_discreta;

    /**
     * @var array|null
     * @Dev5\Orm\Field(['type' => 'json', 'nullable' => true])
     */
    public $categoria;

    /**
     * @var array|null
     * @Dev5\Orm\Field(['type' => 'json', 'nullable' => true])
     */
    public $estoque;

    /**
     * @var array|null
     * @Dev5\Orm\Field(['type' => 'json', 'nullable' => true])
     */
    public $tributacao;

    /**
     * @var array|null
     * @Dev5\Orm\Field(['type' => 'json', 'nullable' => true])
     */
    public $midia;

    /**
     * @var array|null
     * @Dev5\Orm\Field(['type' => 'json', 'nullable' => true])
     */
    public $canais;

    /**
     * @var array|null
     * @Dev5\Orm\Field(['type' => 'json', 'nullable' => true])
     */
    public $campos_customizados;

    /**
     * @param int $find
     * @return static
     */
    public static function getInstance($find = null)
    {
        return parent::getInstance($find ? ['product_id' => $find, 'options' => ''] : null);
    }

    /**
     * @param int $product_id
     * @return array|null
     * @noinspection SqlResolve
     */
    public function getProduct($product_id, $catalog = false)
    {
        if (ocapp('ADMIN') && !$catalog) {
            $data = $this->model_catalog_product->getProduct($product_id);
            if (!$data) {
                return null;
            }

            $class = ocver('4', '<')
                ? "\\ModelCatalogProduct"
                : "\\Opencart\\Admin\\Model\\Catalog\\Product";

            $extra = [
                'product_attribute' => ['getProductAttributes', 'getAttributes'],
                'product_description' => ['getProductDescriptions', 'getDescriptions'],
                'product_discount' => ['getProductDiscounts', 'getDiscounts'],
                'product_filter' => ['getProductFilters', 'getFilters'],
                'product_image' => ['getProductImages', 'getImages'],
                'product_option' => ['getProductOptions', 'getOptions'],
                'product_related' => ['getProductRelated', 'getRelated'],
                'product_reward' => ['getProductRewards', 'getRewards'],
                'product_special' => ['getProductSpecials', 'getSpecials'],
                'product_category' => ['getProductCategories', 'getCategories'],
                'product_download' => ['getProductDownloads', 'getDownloads'],
                'product_layout' => ['getProductLayouts', 'getLayouts'],
                'product_store' => ['getProductStores', 'getStores'],
                'product_recurrings' => ['getRecurrings'],
                'product_subscription' => ['getSubscriptions'],
            ];

            foreach ($extra as $field => $getters) {
                foreach ($getters as $getter) {
                    if (!method_exists($class, $getter)) {
                        continue;
                    }

                    $data[$field] = $this->model_catalog_product->$getter($product_id);
                }

                if ($field === 'product_option') {
                    foreach ($data[$field] as &$product_option) {
                        $product_option['required'] = !!$product_option['required'];
                        if (!empty($product_option['product_option_value'])) {
                            foreach ($product_option['product_option_value'] as &$product_option_value) {
                                $product_option_value['quantity'] = intval($product_option_value['quantity']);
                                $product_option_value['price'] = floatval($product_option_value['price']);
                                $product_option_value['weight'] = floatval($product_option_value['weight']);
                                $product_option_value['points'] = floatval($product_option_value['points']);
                            }
                        }
                    }
                }
            }

            if (ocver('4.1', '<')) {
                $specials = array_filter(isset($data['product_special']) ? $data['product_special'] : [], function ($special) { // phpcs:ignore
                    return $special['customer_group_id'] === $this->config->get('config_customer_group_id')
                        && ($special['date_start'] === '0000-00-00' || strtotime($special['date_start']) < time())
                        && ($special['date_end'] === '0000-00-00' || strtotime($special['date_end']) > time());
                });
            } else {
                $specials = array_map(static function ($special) use ($data) {
                    switch (isset($special['type']) ? $special['type'] : '') {
                        case 'P':
                            $special['price'] = $special['price'] * ($data['price'] / 100);
                            break;

                        case 'S':
                            $special['price'] = $data['price'] - $special['price'];
                            break;
                    }
                    return $special;
                }, array_filter(isset($data['product_discount']) ? $data['product_discount'] : [], static function ($special) { // phpcs:ignore
                    return $special['customer_group_id'] === $this->config->get('config_customer_group_id')
                        && $special['quantity'] == '1'
                        && $special['special'] == '1'
                        && ($special['date_start'] === '0000-00-00' || strtotime($special['date_start']) < time())
                        && ($special['date_end'] === '0000-00-00' || strtotime($special['date_end']) > time());
                }));
            }

            array_multisort(
                array_column($specials, 'priority'),
                array_column($specials, 'price'),
                $specials
            );

            $data['special'] = !empty($specials[0]['price']) ? $specials[0]['price'] : null;

            foreach ($data['product_description'] as &$description) {
                $description['name'] = html_entity_decode($description['name']);
                $description['description'] = html_entity_decode($description['description']);
                $description['meta_title'] = html_entity_decode($description['meta_title']);
                $description['meta_description'] = html_entity_decode($description['meta_description']);
                $description['meta_keyword'] = html_entity_decode($description['meta_keyword']);
            }
        } else {
            $special = (ocver('4.1', '<')
                ? "
                    SELECT
                        price
                    FROM
                        " . DB_PREFIX . "product_special ps
                    WHERE
                        ps.product_id = :product_id
                        AND ps.customer_group_id = :config_customer_group_id
                        AND (
                            (ps.date_start = '0000-00-00' OR ps.date_start < NOW())
                            AND (ps.date_end = '0000-00-00' OR ps.date_end > NOW())
                        )
                    ORDER BY
                        ps.priority,
                        ps.price
                    LIMIT 1
                " : "
                    SELECT
                        (CASE
                            WHEN ps.type = 'P' THEN (ps.price * (p.price / 100))
                            WHEN ps.type = 'S' THEN (p.price - ps.price)
                            ELSE ps.price
                        END)
                    FROM
                        " . DB_PREFIX . "product_discount ps
                    WHERE
                        ps.product_id = :product_id
                        AND ps.customer_group_id = :config_customer_group_id
                        AND ps.quantity = '1'
                        AND ps.special = '1'
                        AND (
                            (ps.date_start = '0000-00-00' OR ps.date_start < NOW())
                            AND (ps.date_end = '0000-00-00' OR ps.date_end > NOW())
                        )
                    ORDER BY
                        ps.priority,
                        ps.price
                    LIMIT 1
            ");

            $data = $this->dev5->db->query("
                SELECT
                    p.sku,
                    pd.name,
                    p.model,
                    p.price,
                    p.status,
                    pd.description,
                    p.weight,
                    p.weight_class_id,
                    m.name AS manufacturer,
                    pd.meta_description,
                    p.length,
                    p.width,
                    p.height,
                    p.length_class_id,
                    p.location,
                    p.image,
                    p.quantity,
                    ($special) AS special,
                    (
                        SELECT
                            r.points
                        FROM
                            " . DB_PREFIX . "product_reward r
                        WHERE
                            r.product_id = :product_id
                            AND r.customer_group_id = :config_customer_group_id
                    ) AS reward
                FROM
                    " . DB_PREFIX . "product p
                INNER JOIN
                    " . DB_PREFIX . "product_description pd
                    ON pd.product_id = p.product_id
                    AND pd.language_id = :language_id
                LEFT JOIN
                    " . DB_PREFIX . "manufacturer m
                    ON p.manufacturer_id = m.manufacturer_id
                WHERE
                    p.product_id = :product_id
            ", [
                'product_id' => $product_id,
                'language_id' => (int)$this->dev5->field('language_id'),
                'config_customer_group_id' => (int)$this->config->get('config_customer_group_id')
            ])->row();

            if (!$data) {
                return null;
            }

            $data['name'] = html_entity_decode($data['name']);

            $data['description'] = html_entity_decode($data['description']);

            $data['meta_description'] = html_entity_decode($data['meta_description']);

            $data['model'] = html_entity_decode($data['model']);

            $data['manufacturer'] = html_entity_decode($data['manufacturer']);

            $data['product_image'] = array_column($this->dev5->db->query("
                SELECT
                    pi.image
                FROM
                    " . DB_PREFIX . "product_image pi
                WHERE
                    pi.product_id = :product_id
            ", [
                'product_id' => $product_id
            ])->all(), 'image');

            $data['product_option'] = array_filter(array_map(function ($product_option) use ($product_id) {

                if (!in_array($product_option['type'], ['radio', 'select', 'checkbox'])) {
                    return null;
                }

                $values = array_map(static function ($product_option_value) {
                    return [
                        'product_option_value_id' => (int)$product_option_value['product_option_value_id'],
                        'sku' => $product_option_value['sku'],
                        'name' => html_entity_decode($product_option_value['name']),
                        'price' => (float)$product_option_value['price'],
                        'price_prefix' => $product_option_value['price_prefix'],
                        'weight' => (float)$product_option_value['weight'],
                        'weight_prefix' => $product_option_value['weight_prefix'],
                        'location' => $product_option_value['location'],
                        'sort_order' => (int)$product_option_value['sort_order'],
                        'quantity' => (int)$product_option_value['quantity']
                    ];
                }, $this->dev5->db->query("
                    SELECT
                        pov.product_option_value_id,
                        ovd.name,
                        {$this->getColumnOptionSKU()} AS sku,
                        pov.price,
                        pov.price_prefix,
                        pov.weight,
                        pov.weight_prefix,
                        {$this->getColumnOptionLocation()} AS location,
                        ov.sort_order,
                        pov.quantity
                    FROM
                        " . DB_PREFIX . "product_option_value pov
                    INNER JOIN
                        " . DB_PREFIX . "option_value ov
                        ON ov.option_value_id = pov.option_value_id
                    INNER JOIN
                        " . DB_PREFIX . "option_value_description ovd
                        ON ovd.option_value_id = pov.option_value_id
                        AND ovd.language_id = :language_id
                    WHERE
                        pov.product_option_id = :product_option_id
                        AND pov.product_id = :product_id
                ", [
                    'product_id' => $product_id,
                    'language_id' => (int)$this->dev5->field('language_id'),
                    'product_option_id' => $product_option['product_option_id']
                ])->all());

                return !$values ? null : [
                    'product_option_id' => (int)$product_option['product_option_id'],
                    'name' => html_entity_decode($product_option['name']),
                    'required' => (int)(bool)$product_option['required'],
                    'sort_order' => (int)$product_option['sort_order'],
                    'type' => $product_option['type'],
                    'values' => $values
                ];
            }, $this->dev5->db->query("
                SELECT
                    po.product_option_id,
                    od.name,
                    po.required,
                    o.type,
                    o.sort_order
                FROM
                    " . DB_PREFIX . "product_option po
                INNER JOIN
                    " . DB_PREFIX . "option o
                    ON o.option_id = po.option_id
                INNER JOIN
                    " . DB_PREFIX . "option_description od
                    ON od.option_id = po.option_id
                WHERE
                    po.product_id = :product_id
                    AND od.language_id = :language_id
            ", [
                'product_id' => $product_id,
                'language_id' => (int)$this->dev5->field('language_id')
            ])->all()));
        }

        $data['product_id'] = intval($product_id);
        $data['price'] = floatval($data['price']);
        $data['special'] = strlen($data['special'] ?: '') ? floatval($data['special']) : null;
        $data['weight'] = floatval($data['weight']);
        $data['length'] = floatval($data['length']);
        $data['width'] = floatval($data['width']);
        $data['height'] = floatval($data['height']);
        $data['weight_class_id'] = intval($data['weight_class_id']);
        $data['length_class_id'] = intval($data['length_class_id']);
        $data['manufacturer'] = html_entity_decode($data['manufacturer']);
        return $data;
    }

    /**
     * @param string $codigo
     * @return array|false
     */
    private function getProductByCode($codigo)
    {
        $rows = $this->dev5->db->querySelect(DB_PREFIX . "product", [
            'columns' => ['product_id'],
            'where' => [$this->dev5->field('code_mode') => $codigo],
            'limit' => 1
        ]);

        if (isset($rows[0]['product_id'])) {
            return $this->getProduct($rows[0]['product_id']);
        }

        return false;
    }

    /**
     * @var array<string, array{
     *     product_id: int,
     *     bling_id: string,
     *     name: string,
     *     is_option: bool
     * }>
     */
    private static $cache_bling_products = [];

    /**
     * @param string $find
     * @param "codigo"|"product_id"|"bling_id" $by
     * @return array|null
     */
    public function getCacheBlingProduct($find, $by = 'product_id')
    {
        switch ($by) {
            case 'codigo':
                return isset(self::$cache_bling_products[$find]) ? self::$cache_bling_products[$find] : null;

            case 'bling_id':
            case 'product_id':
                return array_find(self::$cache_bling_products, static function ($product) use ($by, $find) {
                    return $product[$by] === $find && empty($product['is_option']);
                }) ?: null;
        }

        return null;
    }

    /**
     * @param string $codigo
     * @param array|null $data
     * @return void
     */
    public function setCacheBlingProduct($codigo, $data)
    {
        if (null === $data) {
            unset(self::$cache_bling_products[$codigo]);
        } else {
            self::$cache_bling_products[$codigo] = $data;
        }
    }

    /**
     * @param int $id
     * @param string $sku
     * @param array<array{ id:int, sku:string }>|null $options
     * @return string
     */
    public function makeCode($id, $sku, $options = null)
    {
        static $code_mode = null;
        null === $code_mode && $code_mode = $this->dev5->field('code_mode');

        $code = ($code_mode === 'sku' ? $sku : '') ?: $id;

        if ($options) {
            array_multisort(array_column($options, $code_mode === 'sku' ? 'sku' : 'id'), $options);
            $code .= "-" . implode('-', array_map(function ($option) {
                return $this->makeCode($option['id'], $option['sku']);
            }, $options));
        }

        return $code;
    }

    /**
     * @param array $payload
     * @param string $prefix
     * @return string|false
     */
    public function exportProduct($payload, $prefix = '')
    {
        $this->dev5->debug("$prefix • EXPORT-PRODUCT", $this->product_id);

        $bling_produto = $this->getCacheBlingProduct($this->product_id);

        if ($bling_produto && empty($payload['force'])) {
            $this->dev5->debug("$prefix |→ SUCCESS", "CACHED", $bling_produto['bling_id']);
            return $bling_produto['bling_id'];
        }

        $product = $this->getProduct($this->product_id, true);
        if (!$product) {
            $this->dev5->debug("$prefix |→ FAIL", "NO-DATA");
            return false;
        }

        // phpcs:disable
        $bling_produto = null;

        $product['codigo'] = $this->makeCode($this->product_id, $this->getValue($product, 'sku'));

        if (empty(@$this->dev5->field('ei')['export_product_with_category'])) {
            $category_bling_id = null;
        } elseif ($category_bling_id = $this->getValue($this->categoria, ['id', 'value'])) {
            $category_bling_id = CategoryEntity::getInstance($category_bling_id)->exportCategory($payload, "$prefix |");
        }

        $weight_class_id = intval($this->dev5->field('weight_class_id'));
        $length_class_id = intval($this->dev5->field('length_class_id'));
        $product_weight_class_id = $this->getValue($product, 'weight_class_id');
        $product_length_class_id = $this->getValue($product, 'length_class_id');

        $data = $this->filterNullable([
            'nome' => $this->getValue($product, 'name'),
            'codigo' => $product['codigo'],
            'preco' => $this->currency->convert(floatval($this->getValue($product, 'price')), $this->config->get('config_currency'), $this->dev5->field('currency_code')) ?: null,
            'tipo' => $this->tipo ?: ($this->tipo = 'P'),
            'situacao' => $this->getValue($product, 'status') ? 'A' : 'I',
            'formato' => 'S',
            'descricaoCurta' => $this->getValue($product, 'description') ?: null,
            'dataValidade' => $this->data_validade ? $this->data_validade->format('Y-m-d') : null,
            'unidade' => $this->unidade,
            'pesoLiquido' => $this->weight->convert(floatval($this->peso_liquido), $product_weight_class_id, $weight_class_id) ?: null,
            'pesoBruto' => $this->weight->convert($this->getValue($product, 'weight'), $product_weight_class_id, $weight_class_id) ?: null,
            'volumes' => $this->volumes,
            'itensPorCaixa' => $this->itens_por_caixa,
            'gtin' => $this->gtin,
            'gtinEmbalagem' => $this->gtin_embalagem,
            'tipoProducao' => $this->tipo_producao,
            'condicao' => $this->condicao,
            'freteGratis' => $this->frete_gratis,
            'marca' => $this->getValue($product, 'manufacturer') ?: null,
            'descricaoComplementar' => $this->getValue($product, 'meta_description') ?: null,
            'linkExterno' => substr(str_replace('&amp;', '&', $this->url->link('product/product', 'product_id=' . $this->product_id, ocver('2.2', '<') ? 'SSL' : true)), 0, 120),
            'observacoes' => $this->observacoes,
            'descricaoEmbalagemDiscreta' => $this->descricao_embalagem_discreta,
            'categoria' => $category_bling_id ? ['id' => $category_bling_id] : null,
            'estoque' => $this->filterNullable([
                'minimo' => $this->getValue($this->estoque, 'minimo', 0, 'floatval') ?: null,
                'maximo' => $this->getValue($this->estoque, 'maximo', 0, 'floatval') ?: null,
                'crossdocking' => $this->getValue($this->estoque, 'crossdocking', 0, 'intval') ?: null,
                'localizacao' => $this->getValue($product, 'location') ?: null
            ]),
            // 'actionEstoque' => '',
            'dimensoes' => $this->filterNullable([
                'largura' => $this->length->convert($this->getValue($product, 'width'), $product_length_class_id, $length_class_id) ?: null,
                'altura' => $this->length->convert($this->getValue($product, 'height'), $product_length_class_id, $length_class_id) ?: null,
                'profundidade' => $this->length->convert($this->getValue($product, 'length'), $product_length_class_id, $length_class_id) ?: null,
                'unidadeMedida' => 1
            ]),
            'tributacao' => $this->filterNullable([
                'origem' => $this->getValue($this->tributacao, 'origem', 0, 'intval') ?: null,
                'nFCI' => $this->getValue($this->tributacao, 'nFCI') ?: null,
                'ncm' => $this->getValue($this->tributacao, 'ncm') ?: null,
                'cest' => $this->getValue($this->tributacao, 'cest') ?: null,
                'codigoListaServicos' => $this->getValue($this->tributacao, 'codigoListaServicos') ?: null,
                'spedTipoItem' => $this->getValue($this->tributacao, 'spedTipoItem') ?: null,
                'codigoItem' => $this->getValue($this->tributacao, 'codigoItem') ?: null,
                'percentualTributos' => $this->getValue($this->tributacao, 'percentualTributos', 0, 'floatval') ?: null,
                'valorBaseStRetencao' => $this->getValue($this->tributacao, 'valorBaseStRetencao', 0, 'floatval') ?: null,
                'valorStRetencao' => $this->getValue($this->tributacao, 'valorStRetencao', 0, 'floatval') ?: null,
                'valorICMSSubstituto' => $this->getValue($this->tributacao, 'valorICMSSubstituto', 0, 'floatval') ?: null,
                'codigoExcecaoTipi' => $this->getValue($this->tributacao, 'codigoExcecaoTipi') ?: null,
                'classeEnquadramentoIpi' => $this->getValue($this->tributacao, 'classeEnquadramentoIpi') ?: null,
                'valorIpiFixo' => $this->getValue($this->tributacao, 'valorIpiFixo', 0, 'floatval') ?: null,
                'codigoSeloIpi' => $this->getValue($this->tributacao, 'codigoSeloIpi') ?: null,
                'valorPisFixo' => $this->getValue($this->tributacao, 'valorPisFixo', 0, 'floatval') ?: null,
                'valorCofinsFixo' => $this->getValue($this->tributacao, 'valorCofinsFixo', 0, 'floatval') ?: null,
                'codigoANP' => $this->getValue($this->tributacao, 'codigoANP') ?: null,
                'descricaoANP' => $this->getValue($this->tributacao, 'descricaoANP') ?: null,
                'percentualGLP' => $this->getValue($this->tributacao, 'percentualGLP', 0, 'floatval') ?: null,
                'percentualGasNacional' => $this->getValue($this->tributacao, 'percentualGasNacional', 0, 'floatval') ?: null,
                'percentualGasImportado' => $this->getValue($this->tributacao, 'percentualGasImportado', 0, 'floatval') ?: null,
                'valorPartida' => $this->getValue($this->tributacao, 'valorPartida', 0, 'floatval') ?: null,
                'tipoArmamento' => $this->getValue($this->tributacao, 'tipoArmamento', 0, 'intval') ?: null,
                'descricaoCompletaArmamento' => $this->getValue($this->tributacao, 'descricaoCompletaArmamento') ?: null,
                'dadosAdicionais' => $this->getValue($this->tributacao, 'dadosAdicionais') ?: null,
                'grupoProduto' => ($grupoProdutoId = $this->getValue($this->tributacao, ['grupoProduto', 'id', 'value'])) ? ['id' => $grupoProdutoId] : null
            ]),
            'midia' => $this->filterNullable([
                'video' => $this->filterNullable([
                    'url' => $this->getValue($this->midia, ['video', 'url'])
                ]) ?: null,
                'imagens' => $this->filterNullable([
                    'imagensURL' => $this->filterNullable(
                        array_map(function ($imagem) {
                            if (!$imagem || !file_exists(constant('DIR_IMAGE') . $imagem)) {
                                return null;
                            }

                            static $catalog = null;
                            $catalog === null && $catalog = $this->config->get('config_secure')
                                ? ($this->config->get('config_url') ?: constant('HTTPS_CATALOG'))
                                : ($this->config->get('config_ssl') ?: constant('HTTP_CATALOG'));

                            return ['link' => preg_replace('/(?<!^http:|^https:)\/\/+/', '/', "$catalog/image/$imagem")];
                        }, array_merge(
                            [$this->getValue($product, 'image')],
                            $this->getValue($product, 'product_image', [])
                        ))
                    ) ?: null
                ]) ?: null
            ]),
            'linhaProduto' => null, // ['id' => ?]
            'camposCustomizados' => array_map(static function ($idCampoCustomizado, $value) {
                return ['idCampoCustomizado' => $idCampoCustomizado, 'value' => $value];
            }, array_keys($this->campos_customizados ?: []), array_values($this->campos_customizados ?: [])),
        ]);

        $bycode = [
            $data['codigo'] => [
                'price' => $product['price'],
                'special' => $product['special'],
                'quantity' => $product['quantity']
            ]
        ];

        $variacoes = ($product['options'] = $product['product_option'])
            ? ($this->exportProductNormalizeVariations($data, $product, $bycode) ?: null)
            : null;

        if ($variacoes) {
            $data['formato'] = 'V';
            $data['variacoes'] = $variacoes;
        }
        // phpcs:enable

        if ($this->bling_id && !($bling_produto = Api::getInstance()->getProdutoVariacoes($this->bling_id))) {
            if (!($bling_produto = Api::getInstance()->getProduto($this->bling_id))) {
                $this->bling_id = null;
            }
        }

        if (!$this->bling_id) {
            $temp = @Api::getInstance()->getProdutos(['criterio' => 5, 'codigos' => [$product['codigo']]])[0]['id'];
            !empty($temp) && $bling_produto = Api::getInstance()->getProduto($this->bling_id = $temp);
        }

        if (isset($payload['filter_by_codigo'], $data['variacoes'])) {
            $data['variacoes'] = array_filter($data['variacoes'] ?: [], static function ($variacao) use ($payload) {
                return $variacao['codigo'] === $payload['filter_by_codigo'];
            });
        }

        if (!empty($bling_produto)) {
            $this->exportProductNormalizeImages($bling_produto);

            if (!empty($bling_produto['variacoes'])) {
                if (isset($payload['filter_by_codigo'])) {
                    $bling_produto['variacoes'] = array_filter($bling_produto['variacoes'], static function ($variacao) use ($payload) { // phpcs:ignore
                        return $variacao['codigo'] === $payload['filter_by_codigo'];
                    });
                }

                $this->dev5->debug("$prefix | • CHECKING VARIATIONS");
                $delete_ids = $bling_produto['variacoes'];

                if (!empty($data['variacoes'])) {
                    foreach ($data['variacoes'] as &$variacao) {
                        foreach ($delete_ids as $i => $bling_varicao) {
                            if ($bling_varicao['codigo'] === $variacao['codigo'] || $bling_varicao['variacao']['nome'] === $variacao['variacao']['nome']) { // phpcs:ignore
                                $variacao['id'] = $bling_varicao['id'];
                                unset($delete_ids[$i]);
                                continue 2;
                            }
                        }
                    }
                }

                if ($delete_ids = array_column($delete_ids, 'id')) {
                    $this->dev5->debug("$prefix | | • DELETE VARIATIONS");
                    Api::getInstance()->postProdutosSituacao(['idsProdutos' => $delete_ids, 'situacao' => 'E']);
                    Api::getInstance()->deleteProduto(['idsProdutos' => $delete_ids]);
                    $this->dev5->debug("$prefix | | |→ SUCCESS");
                }

                if ($data['formato'] !== 'V' && $bling_produto['formato'] === 'V') {
                    $this->dev5->debug("$prefix | | • NEED RECREATING", "V -> $data[formato]");
                    Api::getInstance()->postProdutosSituacao(['idsProdutos' => [$this->bling_id], 'situacao' => 'E']);
                    Api::getInstance()->deleteProduto($this->bling_id);
                    $this->bling_id = null;
                    $this->dev5->debug("$prefix | | |→ SUCCESS");
                }
                $this->dev5->debug("$prefix | |→ SUCCESS");
            }
        }

        if (!$this->bling_id) {
            $this->dev5->debug("$prefix | • CREATE");
            $this->bling_id = Api::getInstance()->postProduto($data);
            $bling_produto = null;
        } elseif (array_multidiff($data, $bling_produto)) {
            $this->dev5->debug("$prefix | • UPDATE");
            $this->bling_id = Api::getInstance()->putProduto($this->bling_id, $data);
            $bling_produto = null;
        } else {
            $this->dev5->debug("$prefix | • SKIP", "SAME-DATA");
            $this->bling_id = $bling_produto['id'];
        }

        if (!$this->bling_id) {
            $this->dev5->debug("$prefix | |→ FAIL", Api::getInstance()->error());
            $this->dev5->debug("$prefix |→ FAIL");
            return false;
        }

        $this->options = '';
        $this->codigo = $product['codigo'];

        if (empty($bling_produto) && $bling_produto = Api::getInstance()->getProduto($this->bling_id)) {
            $this->exportProductNormalizeImages($bling_produto);
        }

        if (!$bling_produto) {
            $this->dev5->debug("$prefix | |→ FAIL", Api::getInstance()->error());
            $this->dev5->debug("$prefix |→ FAIL", "BLING-NOT-FOUND");
            return false;
        }

        $this->dev5->debug("$prefix | |→ SUCCESS");

        $bling_ids = [$product['codigo'] => $this->bling_id] + array_column(!empty($bling_produto['variacoes']) ? $bling_produto['variacoes'] : [], 'id', 'codigo'); // phpcs:ignore

        if (!$this->exportProductStock($bling_ids, $bycode, $payload, "$prefix |")) {
            $this->dev5->debug("$prefix |→ FAIL");
            return false;
        }

        if (!$this->exportProductChannels($bling_ids, $bycode, $payload, "$prefix |")) {
            $this->dev5->debug("$prefix |→ FAIL");
            return false;
        }

        ProductEntity::delete(['product_id' => $this->product_id]);

        $this->save(['returning' => false]);
        $this->dev5->debug("$prefix |→ SUCCESS", $this->bling_id);

        $this->setCacheBlingProduct($product['codigo'], [
            'product_id' => $this->product_id,
            'bling_id' => $this->bling_id,
            'codigo' => $product['codigo'],
            'name' => $product['name'],
            'is_option' => false
        ]);

        if (!empty($bling_produto['variacoes'])) {
            foreach ($bling_produto['variacoes'] as $variation) {
                $this->setCacheBlingProduct($variation['codigo'], [
                    'product_id' => $this->product_id,
                    'bling_id' => $variation['id'],
                    'codigo' => $variation['codigo'],
                    'name' => $variation['nome'],
                    'is_option' => true
                ]);

                $variationEntity = new ProductEntity();
                $variationEntity->product_id = $this->product_id;
                $variationEntity->options = $bycode[$variation['codigo']]['options'];
                $variationEntity->bling_id = $variation['id'];
                $variationEntity->bling_master_id = $this->bling_id;
                $variationEntity->codigo = $variation['codigo'];
                $variationEntity->save(['returning' => false]);
            }
        }

        return $this->bling_id;
    }

    /**
     * @param array $bling_product
     * @return void
     */
    private function exportProductNormalizeImages(&$bling_product)
    {
        $bling_product['midia']['imagens']['imagensURL'] = array_merge(
            $this->getValue($bling_product, ['midia', 'imagens', 'imagensURL']) ?: [],
            $this->getValue($bling_product, ['midia', 'imagens', 'externas']) ?: [],
            $this->getValue($bling_product, ['midia', 'imagens', 'internas']) ?: []
        );
        unset($bling_product['midia']['imagens']['externas'], $bling_product['midia']['imagens']['internas']);

        if (!empty($bling_product['variacoes'])) {
            $bling_product['variacoes'] = array_map(function ($variation) {
                $this->exportProductNormalizeImages($variation);
                return $variation;
            }, $bling_product['variacoes']);
        }
    }

    /**
     * @param array $data
     * @param array $product
     * @param array $bycode
     * @return array
     */
    private function exportProductNormalizeVariations($data, &$product, &$bycode)
    {
        if (empty($product['options'])) {
            return [];
        }

        usort($product['options'], static function ($a, $b) {
            $v = $a['product_option_id'] - $b['product_option_id'];
            return $v === 0 ? 0 : ($v > 0 ? 1 : -1);
        });

        foreach ($product['options'] as &$sortOption) {
            usort($sortOption['values'], static function ($a, $b) {
                $v = $a['product_option_value_id'] - $b['product_option_value_id'];
                return $v === 0 ? 0 : ($v > 0 ? 1 : -1);
            });
        }

        $combinations = [];
        $this->exportProductNormalizeVariationsRequired(
            $combinations,
            array_filter($product['options'], static function ($option) {
                return !!$option['required'];
            })
        );
        $this->exportProductNormalizeVariationsOptional(
            $combinations,
            array_filter($product['options'], static function ($option) {
                return !$option['required'];
            }),
            $combinations
        );

        $preco = ($this->getValue($product, 'price', 0, 'floatval') ?: 0);
        $special = $this->getValue($product, 'special', 0, 'floatval');
        $pesoBruto = ($this->getValue($product, 'pesoBruto', 0, 'floatval') ?: 0);
        $pesoLiquido = ($this->getValue($product, 'pesoLiquido', 0, 'floatval') ?: 0);
        $sort_order = 0;

        return array_values(array_map(function ($combination) use ($data, $product, $preco, $special, $pesoBruto, $pesoLiquido, &$sort_order, &$bycode) { // phpcs:ignore
            $variation = $data;

            $variation['codigo'] = $this->makeCode($this->product_id, $product['sku'], $combination['codigo']);

            $bycode[$variation['codigo']] = [
                'price' => $this->currency->convert($preco + $combination['preco'], $this->config->get('config_currency'), $this->dev5->field('currency_code')) ?: null, // phpcs:ignore
                'special' => $special === null ? null : ($this->currency->convert($special + $combination['preco'], $this->config->get('config_currency'), $this->dev5->field('currency_code')) ?: null), // phpcs:ignore
                'quantity' => min($combination['quantidade']),
                'options' => implode(',', $combination['ids'])
            ];

            $variation['nome'] .= ' ' . implode(';', $combination['nome']);
            $variation['preco'] = $bycode[$variation['codigo']]['price'];
            $variation['pesoBruto'] = $this->weight->convert($pesoBruto + $combination['peso'], $product['weight_class_id'], $this->dev5->field('weight_class_id')) ?: null; // phpcs:ignore
            $variation['pesoLiquido'] = $this->weight->convert($pesoLiquido + $combination['peso'], $product['weight_class_id'], $this->dev5->field('weight_class_id')) ?: null; // phpcs:ignore
            $variation['estoque']['localizacao'] = implode(';', array_unique(array_filter($combination['localizacao']))) ?: $product['location'] ?: null; // phpcs:ignore
            $variation['variacao'] = [
                'nome' => implode(';', $combination['nome']),
                'ordem' => ++$sort_order,
                'produtoPai' => ['cloneInfo' => false]
            ];

            return $variation;
        }, $combinations));
    }

    /**
     * @param array $combinations
     * @param array $options
     * @param array|false $combination
     * @return void
     */
    private function exportProductNormalizeVariationsRequired(&$combinations, $options, $combination = false)
    {
        if (!$options) {
            $combination && $combinations[implode('-', $combination['ids'])] = $combination;
            return;
        }

        !$combination && $combination = ['ids' => [], 'codigo' => [], 'nome' => [], 'quantidade' => [], 'preco' => 0, 'peso' => 0, 'localizacao' => []]; // phpcs:ignore

        $option = array_shift($options);
        foreach ($option['values'] as $value) {
            $newCombination = $combination;
            $newCombination['ids'][] = $value['product_option_value_id'];
            $newCombination['codigo'][] = ['id' => $value['product_option_value_id'], 'sku' => $value['sku']];
            $newCombination['nome'][] = "$option[name]:$value[name]";
            $newCombination['quantidade'][] = $value['quantity'];
            $newCombination['preco'] += ($value['price_prefix'] === '-') ? -$value['price'] : $value['price'];
            $newCombination['peso'] += ($value['weight_prefix'] === '-') ? -$value['weight'] : $value['weight'];
            $newCombination['localizacao'][] = $value['location'] ? "$value[location]" : '';
            $this->exportProductNormalizeVariationsRequired($combinations, $options, $newCombination);
        }
    }

    /**
     * @param array $combinations
     * @param array $options
     * @param array $requiredCombinations
     * @return void
     */
    private function exportProductNormalizeVariationsOptional(&$combinations, $options, $requiredCombinations)
    {
        if (!$options) {
            return;
        }

        !$requiredCombinations && $requiredCombinations = ['ids' => [], 'codigo' => [], 'nome' => [], 'quantidade' => [], 'preco' => 0, 'peso' => 0, 'localizacao' => []]; // phpcs:ignore

        $option = array_shift($options);
        foreach ($option['values'] as $value) {
            foreach ($requiredCombinations as $combination) {
                $newCombination = $combination;
                $newCombination['ids'][] = $value['product_option_value_id'];
                $newCombination['codigo'][] = ['id' => $value['product_option_value_id'], 'sku' => $value['sku']];
                $newCombination['nome'][] = "$option[name]:$value[name]";
                $newCombination['quantidade'][] = $value['quantity'];
                $newCombination['preco'] += ($value['price_prefix'] === '-') ? -$value['price'] : $value['price'];
                $newCombination['peso'] += ($value['weight_prefix'] === '-') ? -$value['weight'] : $value['weight'];
                $newCombination['localizacao'][] = $value['location'] ? "$value[location]" : '';
                $combinations[implode('-', $newCombination['ids'])] = $newCombination;

                $this->exportProductNormalizeVariationsOptional($combinations, $options, array_merge($requiredCombinations, [$newCombination])); // phpcs:ignore
            }
        }
    }

    /**
     * @param string[] $bling_ids
     * @param array $bycode
     * @param array $payload
     * @param string $prefix
     * @return bool
     */
    private function exportProductStock($bling_ids, $bycode, $payload, $prefix = '')
    {
        $this->dev5->debug("$prefix • EXPORT-PRODUCT-STOCK", $this->product_id);

        $quantities = array_map(static function ($i) {
            return $i['quantity'];
        }, $bycode);

        if ($payload['_type'] === 'order') {
            // Preventing double stock reduce on order creation
            $this->dev5->debug("$prefix |→ SKIP", "ORDER EVENT");
            return true;
        }

        if (!$this->bling_id) {
            $this->dev5->debug("$prefix |→ FAIL", "NO-DATA");
            return true;
        }

        $this->deposito = $this->deposito ?: $this->dev5->field('deposito');

        $bling_estoques = Api::getInstance()->getEstoquesSaldos(['deposito' => $this->deposito, 'idsProdutos' => $bling_ids]); // phpcs:ignore

        if (!$bling_estoques) {
            $this->dev5->debug("$prefix |→ FAIL", Api::getInstance()->error());
            return false;
        }

        foreach ($bling_estoques as $bling_estoque) {
            $this->dev5->debug("$prefix | • {$bling_estoque['produto']['codigo']}, {$bling_estoque['produto']['id']}");
            if (!array_key_exists($bling_estoque['produto']['codigo'], $quantities)) {
                $this->dev5->debug("$prefix | |→ FAIL", "NO-QUANTITY");
                continue;
            }

            $quentity = floatval($quantities[$bling_estoque['produto']['codigo']]);

            $bling_quantidade = floatval($bling_estoque['saldoFisicoTotal']);

            if ($bling_quantidade === $quentity) {
                $this->dev5->debug("$prefix | |→ SKIP", "SAME-QUANTITY = $bling_quantidade");
                continue;
            }

            if (
                !Api::getInstance()->postEstoque([
                    'produto' => $bling_estoque['produto'],
                    'deposito' => ['id' => $this->deposito],
                    'operacao' => 'B', // B = Balanco, E = Entrada, S = Saida
                    // preco, float
                    // custo, float
                    'quantidade' => floatval($quantities[$bling_estoque['produto']['codigo']]),
                    'observacoes' => 'Balanço via OpenCart'
                ])
            ) {
                $this->dev5->debug("$prefix | |→ FAIL", Api::getInstance()->error());
                $this->dev5->debug("$prefix |→ FAIL");
                return false;
            }

            $this->dev5->debug("$prefix | |→ SUCCESS", "$bling_quantidade => $quentity");
        }

        $this->dev5->debug("$prefix |→ SUCCESS");
        return true;
    }

    /**
     * @param string[] $bling_ids
     * @param array $bycode
     * @param array $payload
     * @param string $prefix
     * @return bool
     */
    private function exportProductChannels($bling_ids, $bycode, $payload, $prefix = '')
    {
        return true;
        $this->dev5->debug("$prefix • EXPORT-PRODUCT-CHANNELS", $this->product_id);

        $prices = array_map(static function ($i) {
            return [
                'price' => $i['price'],
                'special' => $i['special']
            ];
        }, $bycode);

        if (!$this->product_id || !$this->bling_id) {
            $this->dev5->debug("$prefix |→ FAIL", "NO-DATA");
            return true;
        }

        $product_categories = empty(@$this->dev5->field('ei')['export_product_with_category'])
            ? []
            : array_map(static function ($category_id) use ($payload, $prefix) {
                return CategoryEntity::getInstance($category_id)->exportCategory($payload, "$prefix |");
            }, array_column($this->dev5->db->query("
                SELECT
                    `category_id`
                FROM
                    " . DB_PREFIX . "product_to_category
                WHERE
                    `product_id` = :product_id
            ", ['product_id' => $this->product_id])->all(), 'category_id'));

        $channels = [
            $this->dev5->field('channel_id') => array_map(static function ($i) use ($product_categories) {
                $i['categories'] = $product_categories;
                return $i;
            }, $prices)
        ];

        if ($this->canais) {
            foreach ($this->canais as $channel) {
                if (empty($channel['channel_status'])) {
                    continue;
                }

                if (isset($channels[$channel['channel_id']])) {
                    continue;
                }

                $channels[$channel['channel_id']] = array_map(static function ($i) use ($channel) {
                    if ($channel['additional_price']['type'] === 'P') {
                        $i['price'] += $i['price'] * $channel['additional_price']['value'] / 100;
                    } else {
                        $i['price'] += $channel['additional_price']['value'];
                    }

                    if ($i['special'] !== null) {
                        if ($channel['additional_special']['type'] === 'P') {
                            $i['special'] += $i['special'] * $channel['additional_special']['value'] / 100;
                        } else {
                            $i['special'] += $channel['additional_special']['value'];
                        }
                    }

                    return $i;
                }, $prices);
            }
        }

        $bling_canais = [];
        foreach ($bling_ids as $codigo => $bling_id) {
            foreach (Api::getInstance()->getProdutosLojas(['idProduto' => $bling_id]) ?: [] as $bling_canal) {
                $bling_canais[$bling_canal['loja']['id']][$codigo] = [
                    'id' => $bling_canal['id'],
                    'codigo' => $bling_canal['codigo'],
                    'price' => floatval($bling_canal['preco']),
                    'special' => floatval($bling_canal['precoPromocional']),
                    'fornecedor' => $bling_canal['fornecedorLoja']['id'],
                    'marca' => $bling_canal['marcaLoja']['id'],
                    'categories' => array_column($bling_canal['categoriasProdutos'], 'id')
                ];
            }
        }

        $main_channel_id = $this->dev5->field('channel_id');

        foreach ($channels as $channel_id => $products) {
            foreach ($products as $codigo => $channel) {
                $this->dev5->debug("$prefix | • EXPORT-PRODUCT-CHANNEL", "$codigo: $channel_id");
                $channel['special'] === null && $channel['special'] = $channel['price'];

                $channel['price'] = round($channel['price'], 2);
                $channel['special'] = round($channel['special'], 2);

                if (!isset($bling_canais[$channel_id][$codigo])) {
                    if ($main_channel_id != $channel_id) {
                        // Lojas diferentes da principal é necessário criar manualmente no Bling
                        $this->dev5->debug("$prefix | |→ SKIP", "NOT-CREATE-IF-NOT-MAIN-CHANNEL");
                        continue;
                    }

                    $this->dev5->debug("$prefix | | • CREATE");
                    $id_vinculo = Api::getInstance()->postProdutoLoja([
                        'codigo' => $codigo,
                        'preco' => $channel['price'],
                        'precoPromocional' => $channel['special'],
                        'produto' => ['id' => $bling_ids[$codigo]],
                        'loja' => ['id' => $channel_id],
                        'categoriasProdutos' => array_map(static function ($id) {
                            return ['id' => $id];
                        }, $channel['categories'])
                    ]);

                    if (!$id_vinculo) {
                        $this->dev5->debug("$prefix | | |→ FAIL", Api::getInstance()->error());
                        $this->dev5->debug("$prefix | |→ FAIL");
                        $this->dev5->debug("$prefix |→ FAIL");
                        return false;
                    }

                    $this->dev5->debug("$prefix | |→ SUCCESS", $id_vinculo);
                    continue;
                }

                $bling_canal = $bling_canais[$channel_id][$codigo];

                $compare = $channel['price'] === $bling_canal['price']
                    && $channel['special'] === $bling_canal['special'];

                if ($compare) {
                    $this->dev5->debug("$prefix | |→ SKIP", "SAME-DATA", "$bling_canal[id]");
                    continue;
                }

                $this->dev5->debug("$prefix | | • UPDATE");
                $result = Api::getInstance()->putProdutoLoja($bling_canal['id'], [
                    'codigo' => $main_channel_id != $channel_id ? $bling_canal['codigo'] : $codigo,
                    'preco' => $channel['price'],
                    'precoPromocional' => $channel['special'],
                    'produto' => ['id' => $bling_ids[$codigo]],
                    'loja' => ['id' => $channel_id],
                    'fornecedorLoja' => ['id' => $bling_canal['fornecedor']],
                    'marcaLoja' => ['id' => $bling_canal['marca']],
                    'categoriasProdutos' => array_map(static function ($id) {
                        return ['id' => $id];
                    }, $main_channel_id != $channel_id ? $bling_canal['categories'] : $channel['categories'])
                ]);

                if (!$result) {
                    $this->dev5->debug("$prefix | | |→ FAIL", Api::getInstance()->error());
                    $this->dev5->debug("$prefix | |→ FAIL");
                    $this->dev5->debug("$prefix |→ FAIL");
                    return false;
                }

                $this->dev5->debug("$prefix | | |→ SUCCESS", $bling_canal['id']);
                $this->dev5->debug("$prefix | |→ SUCCESS");
            }
        }

        $this->dev5->debug("$prefix |→ SUCCESS");
        return true;
    }

    /**
     * @param array $payload
     * @param string $prefix
     * @return bool
     */
    public function exportDeleteProduct($payload, $prefix = '')
    {
        $this->dev5->debug("$prefix • DELETE-PRODUCT", $this->product_id);

        if (!$this->bling_id) {
            $this->dev5->debug("$prefix |→ SUCCESS", "NO-DATA");
            return true;
        }

        $bling_product = Api::getInstance()->getProdutoVariacoes($this->bling_id);
        $bling_ids = [$this->bling_id] + array_column(!empty($bling_product['variacoes']) ? $bling_product['variacoes'] : [], 'id'); // phpcs:ignore

        if (!Api::getInstance()->postProdutosSituacao(['idsProdutos' => $bling_ids, 'situacao' => 'E'])) {
            $this->dev5->debug("$prefix |→ FAIL", Api::getInstance()->error());
            return false;
        }

        if (!Api::getInstance()->deleteProduto($bling_ids)) {
            $this->dev5->debug("$prefix |→ FAIL", Api::getInstance()->error());
            return false;
        }

        ProductEntity::delete($this->product_id);
        foreach ($bling_ids as $bling_id) {
            if ($cache = $this->getCacheBlingProduct($bling_id, 'bling_id')) {
                $this->setCacheBlingProduct($cache['codigo'], null);
            }
        }

        $this->dev5->debug("$prefix |→ SUCCESS");
        return true;
    }

    /**
     * @param EIEntity\Payload $payload
     * @param string $prefix
     * @return bool
     */
    public function exportProducts($payload, $prefix = '')
    {
        $this->dev5->debug("$prefix • EXPORT-PRODUCTS", $payload->cursor);

        $where = [];
        $values = ['language_id' => (int)$this->dev5->field('language_id')];

        if (!empty($payload->input['id']) || !empty($payload->input['name'])) {
            if (!empty($payload->input['id'])) {
                $product_ids = $payload->input['id'];
            }

            if (!empty($payload->input['name'])) {
                $temp_ids = array_column($payload->input['name'], 'value');
                $product_ids = isset($product_ids)
                    ? array_intersect($product_ids, $temp_ids)
                    : $temp_ids;
            }

            $where[] = "p.product_id IN :product_id";
            $values['product_id'] = empty($product_ids) ? [-1] : array_unique($product_ids);
        }

        if (array_key_exists('status', $payload->input) && strlen($payload->input['status']) > 0) {
            $where[] = "p.status = :status";
            $values['status'] = $payload->input['status'];
        }

        if (!empty($payload->input['date_added']['start']) && !empty($payload->input['date_added']['end'])) {
            $where[] = "p.date_added BETWEEN :date_added_start AND :date_added_end";
            $values['date_added_start'] = "{$payload->input['date_added']['start']} 00:00:00";
            $values['date_added_end'] = "{$payload->input['date_added']['end']} 23:59:59";
        } elseif (!empty($payload->input['date_added']['start'])) {
            $where[] = "p.date_added >= :date_added_start";
            $values['date_added_start'] = "{$payload->input['date_added']['start']} 00:00:00";
        } elseif (!empty($payload->input['date_added']['end'])) {
            $where[] = "p.date_added <= :date_added_end";
            $values['date_added_end'] = "{$payload->input['date_added']['end']} 23:59:59";
        }

        if (!empty($payload->input['date_modified']['start']) && !empty($payload->input['date_modified']['end'])) {
            $where[] = "p.date_modified BETWEEN :date_modified_start AND :date_modified_end";
            $values['date_modified_start'] = "{$payload->input['date_modified']['start']} 00:00:00";
            $values['date_modified_end'] = "{$payload->input['date_modified']['end']} 23:59:59";
        } elseif (!empty($payload->input['date_modified']['start'])) {
            $where[] = "p.date_modified >= :date_modified_start";
            $values['date_modified_start'] = "{$payload->input['date_modified']['start']} 00:00:00";
        } elseif (!empty($payload->input['date_modified']['end'])) {
            $where[] = "p.date_modified <= :date_modified_end";
            $values['date_modified_end'] = "{$payload->input['date_modified']['end']} 23:59:59";
        }

        $where = empty($where) ? '' : 'WHERE ' . implode(' AND ', $where);
        $limit = 100;
        $offset = $payload->cursor * $limit;

        $rows = $this->dev5->db->query("
            SELECT
                p.product_id,
                pd.name
            FROM
                " . DB_PREFIX . "product p
            INNER JOIN
                " . DB_PREFIX . "product_description pd
                ON pd.product_id = p.product_id
                AND pd.language_id = :language_id
            $where
            ORDER BY p.product_id
            LIMIT $offset, $limit;
        ", $values)->all();

        foreach ($rows as $row) {
            EIEntity::fromArray([
                'id' => $row['product_id'],
                'label' => "#$row[product_id] $row[name]"
            ])->save(['returning' => false]);
            $payload->total++;

            $this->dev5->debug("$prefix | • $row[product_id] → PENDING");
        }

        $count = count($rows);

        $this->dev5->debug("$prefix |→ SUCCESS", "$count/$limit");

        return $count < $limit;
    }

    /**
     * @param EIEntity\Payload $payload
     * @param string $prefix
     * @return bool
     */
    public function exportProductsApprove($payload, $prefix = '')
    {
        $this->dev5->debug("$prefix • EXPORT-PRODUCTS-APPROVE", $payload->cursor);

        $limit = 100;
        $entities = EIEntity::findMany(['status' => EIEntity::PENDING], ['limit' => $limit]);

        foreach ($entities as $entity) {
            $this->dev5->debug("$prefix • EXPORT-PRODUCT-APPROVE", $entity->id);

            QueueEntity::publish("export_product:$entity->id", ['event' => 'ei', 'force' => !empty($payload->input['force'])]); // phpcs:ignore
            $payload->total++;

            $entity->status = EIEntity::QUEUE;
            $entity->save(['returning' => false]);

            $this->dev5->debug("$prefix | |→ QUEUE");
        }

        $count = count($entities);

        $this->dev5->debug("$prefix |→ SUCCESS", "$count/$limit");

        return $count < $limit;
    }

    /**
     * @param string $bling_id
     * @param EIEntity\Payload|null $payload
     * @param string $prefix
     * @return false|int|mixed
     * @throws Exception
     */
    public function importProduct($bling_id, $payload = null, $prefix = '', &$error = null)
    {
        $this->dev5->debug("$prefix • IMPORT-PRODUCT", "$bling_id");

        if ($bling_product = $this->getCacheBlingProduct($bling_id, 'bling_id')) {
            $this->dev5->debug("$prefix |→ SUCCESS", "CACHED", $bling_product['product_id']);
            return $bling_product['product_id'];
        }

        $bling_product = Api::getInstance()->getProduto($bling_id);
        if (!$bling_product) {
            $error = ['message' => 'FAIL, NO-DATA', 'file' => __FILE__ . ':' . __LINE__, 'debug' => Api::getInstance()->last()]; // phpcs:ignore
            $this->dev5->debug("$prefix |→ FAIL", "NO-DATA", Api::getInstance()->error());
            return false;
        }

        if (!empty($bling_product['variacao']['produtoPai']['id'])) {
            $this->dev5->debug("$prefix | • IMPORT-PRODUCT-OPTION-PARENT", $bling_product['variacao']['produtoPai']['id']); // phpcs:ignore

            $parent_id = $this->importProduct($bling_product['variacao']['produtoPai']['id'], $payload, "$prefix | |", $e); // phpcs:ignore
            if (!$parent_id) {
                $this->dev5->debug("$prefix | |→ FAIL");
                $this->dev5->debug("$prefix |→ FAIL");
                return false;
            }

            $this->dev5->debug("$prefix | |→ SUCCESS");
            $this->dev5->debug("$prefix |→ SUCCESS", $parent_id);
            return $parent_id;
        }

        $entity = self::findOne(['bling_id' => $bling_id]) ?: new self();
        $entity->bling_id = $bling_id;

        if ($entity->product_id && (!($current = $this->getProduct($entity->product_id)))) {
            $entity->product_id = 0;
        }

        if (!$entity->product_id && ($current = $this->getProductByCode($bling_product['codigo']))) {
            $entity->product_id = $current['product_id'];
        }

        // phpcs:disable
        $current = empty($current) ? [] : $current;

        $data = $current;
        $data['master_id'] = $this->getValue($data, 'master_id', 0);
        $data['model'] = $this->getValue($data, 'model', '');
        $data['sku'] = ($this->dev5->field('code_mode') === 'sku')
            ? $this->getValue($bling_product, 'codigo', $entity->codigo)
            : $this->getValue($data, 'sku', $entity->codigo);
        $data['upc'] = $this->getValue($data, 'upc', '');
        $data['ean'] = $this->getValue($data, 'ean', '');
        $data['jan'] = $this->getValue($data, 'jan', '');
        $data['isbn'] = $this->getValue($data, 'isbn', '');
        $data['mpn'] = $this->getValue($data, 'mpn', '');
        $data['location'] = $this->getValue($bling_product, ['estoque', 'localizacao'], '');
        $data['variant'] = $this->getValue($data, 'variant', []);
        $data['override'] = $this->getValue($data, 'override', []);
        $data['quantity'] = $this->getValue($bling_product, ['estoque', 'saldoVirtualTotal'], 0, 'intval');
        $data['minimum'] = $this->getValue($data, 'minimum', 1);
        $data['subtract'] = (int)$this->getValue($data, 'subtract', 1);
        $data['stock_status_id'] = $this->getValue($data, 'stock_status_id', 0);
        $data['date_available'] = $this->getValue($data, 'date_available', date('Y-m-d'));
        $data['manufacturer_id'] = $this->getValue($data, 'manufacturer_id', 0);
        $data['shipping'] = 1;
        $data['price'] = $this->getValue($bling_product, 'preco', 0, 'floatval');
        $data['points'] = $this->getValue($data, 'points', 0);
        $data['weight'] = $this->getValue($bling_product, 'pesoBruto', 0, 'floatval');
        $data['weight_class_id'] = $this->dev5->field('weight_class_id');
        $data['length'] = $this->getValue($bling_product, ['dimensoes', 'profundidade'], 0, 'floatval');
        $data['width'] = $this->getValue($bling_product, ['dimensoes', 'largura'], 0, 'floatval');
        $data['height'] = $this->getValue($bling_product, ['dimensoes', 'altura'], 0, 'floatval');
        $data['length_class_id'] = $this->dev5->field('length_class_id');
        $data['status'] = $this->getValue($bling_product, ['situacao'], 'A') === 'A' ? 1 : 0;
        $data['tax_class_id'] = $this->getValue($data, 'tax_class_id', 0);
        $data['sort_order'] = $this->getValue($data, 'sort_order', 0);
        unset($data['image']); // importProductImage
        foreach ($this->getLanguages() as $language_id) {
            $data['product_description'][$language_id]['name'] = $this->getValue($bling_product, 'nome');
            $data['product_description'][$language_id]['description'] = $this->getValue($bling_product, 'descricaoCurta');
            $data['product_description'][$language_id]['tag'] = $this->getValue($data, ['product_description', $language_id, 'tag'], '');
            $data['product_description'][$language_id]['meta_title'] = $this->getValue($bling_product, 'nome');
            $data['product_description'][$language_id]['meta_description'] = $this->getValue($bling_product, 'descricaoComplementar');
            $data['product_description'][$language_id]['meta_keyword'] = $this->getValue($data, ['product_description', $language_id, 'meta_keyword'], '');
        }
        $data['product_category'] = $this->getValue($data, 'product_category', []);
        $data['product_filter'] = $this->getValue($data, 'product_filter', []);
        $data['product_store'] = $this->getValue($data, 'product_store', array_merge([0], array_column($this->model_setting_store->getStores(), 'store_id')));
        $data['product_download'] = $this->getValue($data, 'product_download', []);
        $data['product_related'] = $this->getValue($data, 'product_related', []);
        $data['product_attribute'] = $this->getValue($data, 'product_attribute', []);
        $this->importProductOptions($data, $current, $bling_product); // $data['product_option']
        $data['product_subscription'] = $this->getValue($data, 'product_subscription', []);
        $data['product_discount'] = $this->getValue($data, 'product_discount', []);
        $data['product_special'] = $this->getValue($data, 'product_special', []);
        unset($data['product_image']); // importProductImage
        $data['product_reward'] = $this->getValue($data, 'product_reward', []);
        $data['product_layout'] = $this->getValue($data, 'product_layout', []);
        $data['keyword'] = $this->getValue($data, 'keyword', '');
        $data['product_seo_url'] = $this->getValue($data, 'product_seo_url', []);
        $data['product_recurrings'] = $this->getValue($data, 'product_recurrings', []);
        $data['special'] = $this->getValue($data, 'special', 0, 'floatval');
        $this->importProductImage($data, $bling_product);

        $entity->bling_id = $bling_id;
        $entity->options = '';
        $entity->bling_master_id = '';
        $entity->codigo = $this->getValue($bling_product, 'codigo', '');
        $entity->tipo = $this->getValue($bling_product, 'tipo', 'P');
        $entity->data_validade = $this->getValue($bling_product, 'dataValidade', '') ?: null;
        $entity->unidade = $this->getValue($bling_product, 'unidade', '') ?: null;
        $entity->deposito = $this->getValue($bling_product, 'deposito', $this->dev5->field('deposito'));
        $entity->peso_liquido = $this->getValue($bling_product, 'pesoLiquido', 0, 'floatval') ?: null;
        $entity->volumes = $this->getValue($bling_product, 'volumes', 0, 'intval') ?: null;
        $entity->itens_por_caixa = $this->getValue($bling_product, 'itensPorCaixa', 0, 'intval') ?: null;
        $entity->gtin = $this->getValue($bling_product, 'gtin', '') ?: null;
        $entity->gtin_embalagem = $this->getValue($bling_product, 'gtinEmbalagem', '') ?: null;
        $entity->tipo_producao = $this->getValue($bling_product, 'tipoProducao', '') ?: null;
        $entity->condicao = $this->getValue($bling_product, 'condicao', '') ?: null;
        $entity->frete_gratis = $this->getValue($bling_product, 'freteGratis', 0, 'intval');
        $entity->observacoes = $this->getValue($bling_product, 'observacoes', '') ?: null;
        $entity->descricao_embalagem_discreta = $this->getValue($bling_product, 'descricaoEmbalagemDiscreta', '') ?: null;
        $entity->categoria = ['id' => $this->importProductCategoryAutocomplete($bling_product, $payload, "$prefix |")];
        $entity->estoque = [
            'minimo' => $this->getValue($bling_product, ['estoque', 'minimo'], 0, 'intval'),
            'maximo' => $this->getValue($bling_product, ['estoque', 'maximo'], 0, 'intval'),
            'crossdocking' => $this->getValue($bling_product, ['estoque', 'crossdocking'], 0, 'intval'),
        ];
        $entity->tributacao = [
            'ncm' => $this->getValue($bling_product, ['tributacao', 'ncm'], ''),
            'cest' => $this->getValue($bling_product, ['tributacao', 'cest'], ''),
            'nFCI' => $this->getValue($bling_product, ['tributacao', 'nFCI'], ''),
            'origem' => $this->getValue($bling_product, ['tributacao', 'origem'], ''),
            'codigoANP' => $this->getValue($bling_product, ['tributacao', 'codigoANP'], ''),
            'codigoItem' => $this->getValue($bling_product, ['tributacao', 'codigoItem'], ''),
            'descricaoANP' => $this->getValue($bling_product, ['tributacao', 'descricaoANP'], ''),
            'grupoProduto' => ['id' => $this->importProductGroupProductAutocomplete($bling_product)],
            'spedTipoItem' => $this->getValue($bling_product, ['tributacao', 'spedTipoItem'], ''),
            'valorIpiFixo' => $this->getValue($bling_product, ['tributacao', 'valorIpiFixo'], 0.0, 'floatval'),
            'valorPartida' => $this->getValue($bling_product, ['tributacao', 'valorPartida'], 0.0, 'floatval'),
            'valorPisFixo' => $this->getValue($bling_product, ['tributacao', 'valorPisFixo'], 0.0, 'floatval'),
            'codigoSeloIpi' => $this->getValue($bling_product, ['tributacao', 'codigoSeloIpi'], ''),
            'percentualGLP' => $this->getValue($bling_product, ['tributacao', 'percentualGLP'], 0.0, 'floatval'),
            'tipoArmamento' => $this->getValue($bling_product, ['tributacao', 'tipoArmamento'], ''),
            'dadosAdicionais' => $this->getValue($bling_product, ['tributacao', 'dadosAdicionais'], ''),
            'valorCofinsFixo' => $this->getValue($bling_product, ['tributacao', 'valorCofinsFixo'], 0.0, 'floatval'),
            'valorStRetencao' => $this->getValue($bling_product, ['tributacao', 'valorStRetencao'], 0.0, 'floatval'),
            'codigoExcecaoTipi' => $this->getValue($bling_product, ['tributacao', 'codigoExcecaoTipi'], ''),
            'percentualTributos' => $this->getValue($bling_product, ['tributacao', 'percentualTributos'], 0.0, 'floatval'),
            'codigoListaServicos' => $this->getValue($bling_product, ['tributacao', 'codigoListaServicos'], ''),
            'valorBaseStRetencao' => $this->getValue($bling_product, ['tributacao', 'valorBaseStRetencao'], 0.0, 'floatval'),
            'valorICMSSubstituto' => $this->getValue($bling_product, ['tributacao', 'valorICMSSubstituto'], 0.0, 'floatval'),
            'percentualGasNacional' => $this->getValue($bling_product, ['tributacao', 'percentualGasNacional'], 0.0, 'floatval'),
            'classeEnquadramentoIpi' => $this->getValue($bling_product, ['tributacao', 'classeEnquadramentoIpi'], ''),
            'percentualGasImportado' => $this->getValue($bling_product, ['tributacao', 'percentualGasImportado'], 0.0, 'floatval'),
            'descricaoCompletaArmamento' => $this->getValue($bling_product, ['tributacao', 'descricaoCompletaArmamento'], ''),
        ];
        $entity->midia = [
            'video' => ['url' => $this->getValue($bling_product, ['midia', 'video', 'url'], '')]
        ];
        $entity->campos_customizados = array_column(array_map(function ($campo_customizado) {
            return [
                'id' => $this->getValue($campo_customizado, 'id'),
                'value' => $this->getValue($campo_customizado, 'valor')
            ];
        }, $this->getValue($bling_product, 'camposCustomizados', [])), 'valor', 'id');
        $entity->canais = array_map(function ($channel) use (&$data, $payload, $prefix) {
            $channel_id = $this->getValue($channel, ['loja', 'id']);
            $preco = $this->getValue($channel, 'preco', 0, 'floatval');
            $precoPromocional = $this->getValue($channel, 'precoPromocional', 0, 'floatval');

            if ($data['price']) {
                $additional_price = ['type' => 'P', 'value' => $preco ? (($preco * 100 / $data['price'] - 100) ?: '') : ''];
            } else {
                $additional_price = ['type' => '', 'value' => ($data['price'] - $preco) ?: ''];
            }

            if ($data['special']) {
                $additional_special = ['type' => 'P', 'value' => $precoPromocional ? (($precoPromocional * 100 / $data['special'] - 100) ?: '') : ''];
            } else {
                $additional_special = ['type' => '', 'value' => ($data['special'] - $precoPromocional) ?: ''];
            }

            if ($channel_id == $this->dev5->field('channel_id')) {
                $data['product_category'] = [];
                if (!empty($channel['categoriasProdutos'])) {
                    foreach ($channel['categoriasProdutos'] as $categoria) {
                        static $categoria_produtos_loja = [];
                        if (!isset($categoria_produtos_loja[$categoria['id']])) {
                            $categoria_produtos_loja[$categoria['id']] = $this->getValue(
                                Api::getInstance()->getCategoriaProdutoLoja($categoria['id']), ['categoriaProduto', 'id']
                            );
                        }

                        if (!isset($categoria_produtos_loja[$categoria['id']])) {
                            continue;
                        }

                        if ($category_id = CategoryEntity::getInstance()->importCategory($categoria_produtos_loja[$categoria['id']], $payload, "$prefix |")) {
                            $data['product_category'][] = $category_id;
                        }
                    }
                }
            }

            return [
                'channel_id' => $channel_id,
                'channel_status' => 1,
                'additional_price' => $additional_price,
                'additional_special' => $additional_special
            ];
        }, Api::getInstance()->getProdutosLojas(['idProduto' => $bling_id]) ?: []);
        // phpcs:enable

        try {
            $compare = $data;
            $compare['product_image'] = array_map(static function ($product_image) {
                unset($product_image['sort_order']);
                return $product_image;
            }, $compare['product_image']);
            unset($compare[ocver('3', '>=') ? 'keyword' : 'product_seo_url']);
            if (ocver('4', '<')) {
                unset($compare['override']);
                unset($compare['master_id']);
                unset($compare['variant']);
                unset($compare['product_subscription']);
            }

            self::ignore(true);
            if (!array_multidiff($compare, $current)) {
                $this->dev5->debug("$prefix | • SKIP", "SAME DATA");
            } elseif (!$entity->product_id) {
                $this->dev5->debug("$prefix | • CREATE");
                $entity->product_id = $this->model_catalog_product->addProduct($data);
            } else {
                $this->dev5->debug("$prefix | • UPDATE");
                $this->model_catalog_product->editProduct($entity->product_id, $data);
            }
        } catch (Exception $e) {
            $error = "$e";
            $this->dev5->debug("$prefix | |→ FAIL", "$e");
            $this->dev5->debug("$prefix |→ FAIL");
            return false;
        } catch (Throwable $e) {
            $error = "$e";
            $this->dev5->debug("$prefix | |→ FAIL", "$e");
            $this->dev5->debug("$prefix |→ FAIL");
            return false;
        } finally {
            self::ignore(false);
        }

        $this->setCacheBlingProduct($bling_product['codigo'], [
            'product_id' => $entity->product_id,
            'bling_id' => $bling_id,
            'name' => $bling_product['nome'],
            'is_option' => false
        ]);
        $entity->save(['returning' => false]);

        if (!empty($bling_product['variacoes'])) {
            foreach ($bling_product['variacoes'] as $variacao) {
                $this->setCacheBlingProduct($variacao['codigo'], [
                    'product_id' => $entity->product_id,
                    'bling_id' => $variacao['id'],
                    'name' => $variacao['nome'],
                    'is_option' => true
                ]);

                $variationEntity = new ProductEntity();
                $variationEntity->product_id = $entity->product_id;
                $variationEntity->options = implode(',', $this->importProductVariationToProductOptions($entity->product_id, $variacao)); // phpcs:ignore
                $variationEntity->bling_id = $variacao['id'];
                $variationEntity->bling_master_id = $this->bling_id;
                $variationEntity->codigo = $variacao['codigo'];
                $variationEntity->save(['returning' => false]);
            }
        }
        $this->dev5->debug("$prefix | |→ SUCCESS");
        $this->dev5->debug("$prefix |→ SUCCESS", $entity->product_id);
        return $entity->product_id;
    }

    /**
     * @param array $data
     * @param array $bling_product
     * @return void
     */
    private function importProductImage(&$data, $bling_product)
    {
        $urls = array_column(array_merge(
            $this->getValue($bling_product, ['midia', 'imagens', 'externas'], []),
            $this->getValue($bling_product, ['midia', 'imagens', 'internas'], [])
        ), 'link');

        $data['image'] = '';
        $data['product_image'] = [];
        $sort_order = -1;

        foreach ($urls as $url) {
            if (!($image = $this->importProductImageDownload($url))) {
                continue;
            }

            if ($sort_order === -1) {
                $data['image'] = $image;
            } else {
                $data['product_image'][] = ['image' => $image, 'sort_order' => $sort_order];
            }

            $sort_order++;
        }
    }

    /**
     * @param string $url
     * @return string|false
     */
    private function importProductImageDownload($url)
    {
        static $catalog = null;
        $catalog === null && $catalog = parse_url(preg_replace(
            '/(?<!^http:|^https:)\/\/+/',
            '/',
            ($this->config->get('config_secure')
                ? ($this->config->get('config_url') ?: constant('HTTPS_CATALOG'))
                : ($this->config->get('config_ssl') ?: constant('HTTP_CATALOG'))
            ) . '/image/'
        ), PHP_URL_PATH) ?: '/';

        $image = parse_url($url, PHP_URL_PATH);

        if (!$image) {
            return false;
        }

        if (is_file(DIR_IMAGE . substr($image, strlen($catalog)))) {
            return substr($image, strlen($catalog));
        }

        $image = 'catalog/d5bling/' . sha1($image);
        foreach (glob(DIR_IMAGE . "$image.*", GLOB_BRACE) as $file) {
            return $image . '.' . pathinfo($file, PATHINFO_EXTENSION);
        }

        $temp = DIR_IMAGE . "$image~";
        try {
            is_dir(dirname($temp)) || @mkdir(dirname($temp), 0755, true);
            is_file($temp) && @unlink($temp);
            if (!($fp = fopen($temp, 'w'))) {
                return false;
            }

            $response = Http::get($url, [], 10, [CURLOPT_FILE => $fp]);
        } finally {
            @fclose($fp);
        }

        if ($response->code !== 200 || !str_starts_with($response->info['content_type'], 'image/')) {
            @unlink($temp);
            return false;
        }

        $image .= '.' . substr($response->info['content_type'], 6); // extension

        is_file(DIR_IMAGE . $image) && @unlink(DIR_IMAGE . $image);
        @rename($temp, DIR_IMAGE . $image);

        return $image;
    }

    /**
     * @param array $data
     * @param array $current
     * @param array $bling_produto
     * @return void
     */
    private function importProductOptions(&$data, $current, $bling_produto)
    {
        empty($current['product_option']) && $current['product_option'] = [];

        $language_id = (int)$this->dev5->field('language_id');

        $product_options = [];

        if (!empty($current['product_option'])) {
            foreach ($current['product_option'] as $product_option) {
                if (!empty($product_option['product_option_value'])) {
                    continue;
                }

                $product_options[] = $product_option;
            }
        }

        if ($variacoes = $this->getValue($bling_produto, 'variacoes') ?: []) {
            $count = 0;
            $quantity = 0;
            $product_options_prices = $this->importProductOptionsPrices($data, $bling_produto);

            foreach ($variacoes as $variacao) {
                $count++;

                $options = array_column(array_map(static function ($i) {
                    return explode(':', $i);
                }, explode(';', $this->getValue($variacao, ['variacao', 'nome']) ?: [])), 1, 0);
                foreach ($options as $name => $value) {
                    static $cache_product_option = [];
                    if (!$this->getValue($cache_product_option, $name)) {
                        $cache_product_option[$name] = array_merge(['values' => []], $this->dev5->db->query("
                            SELECT
                                o.option_id,
                                o.type
                            FROM
                                `" . DB_PREFIX . "option` o
                            INNER JOIN
                                `" . DB_PREFIX . "option_description` od
                                ON o.option_id = od.option_id
                                AND od.language_id = :language_id
                            WHERE
                                od.name = :name
                        ", [
                            'language_id' => $language_id,
                            'name' => $name
                        ])->row() ?: ['option_id' => 0, 'type' => 'select']);
                    }
                    if (!$this->getValue($cache_product_option, [$name, 'values', $value]) && $this->getValue($cache_product_option, [$name, 'option_id'])) { // phpcs:ignore
                        $cache_product_option[$name]['values'][$value] = $this->dev5->db->query("
                            SELECT
                                ov.option_value_id
                            FROM
                                `" . DB_PREFIX . "option_value` ov
                            INNER JOIN
                                `" . DB_PREFIX . "option_value_description` ovd
                                ON ov.option_value_id = ovd.option_value_id
                                AND ovd.language_id = :language_id
                            WHERE
                                ov.option_id = :option_id
                                AND ovd.name = :name
                        ", [
                            'language_id' => $language_id,
                            'option_id' => $cache_product_option[$name]['option_id'],
                            'name' => $value
                        ])->column();
                    }

                    $price = $product_options_prices["$name:$value"];
                    $price_prefix = '+';
                    if ($price < 0) {
                        $price = $price * -1;
                        $price_prefix = '-';
                    }

                    $current_option = null;
                    if ($this->getValue($cache_product_option, [$name, 'values', $value])) {
                        foreach ($current['product_option'] as $product_option) {
                            if ($product_option['option_id'] == $this->getValue($cache_product_option, [$name, 'option_id'])) { // phpcs:ignore
                                foreach ($product_option['product_option_value'] as $product_option_value) {
                                    if ($product_option_value['option_value_id'] == $this->getValue($cache_product_option, [$name, 'values', $value])) { // phpcs:ignore
                                        $current_option = [
                                            'product_option_id' => $product_option['product_option_id'],
                                            'product_option_value_id' => $product_option_value['product_option_value_id'],
                                            'quantity' => $product_option_value['quantity'],
                                            'price' => $price,
                                            'price_prefix' => $price_prefix,
                                            'points' => $product_option_value['points'],
                                            'points_prefix' => $product_option_value['points_prefix'],
                                            'weight' => $product_option_value['weight'],
                                            'weight_prefix' => $product_option_value['weight_prefix'],
                                        ];
                                        break 2;
                                    }
                                }
                            }
                        }
                    }

                    if (null === $current_option) {
                        if (!($option_id = $this->getValue($cache_product_option, [$name, 'option_id']))) {
                            $option_id = $this->dev5->db->query("
                                INSERT INTO
                                    `" . DB_PREFIX . "option`
                                SET
                                    `type` = 'select',
                                    `sort_order` = 0
                            ")->lastId();
                            foreach ($this->getLanguages() as $language_id) {
                                $this->dev5->db->query("
                                    INSERT INTO
                                        `" . DB_PREFIX . "option_description`
                                    SET
                                        `option_id` = :option_id,
                                        `language_id` = :language_id,
                                        `name` = :name
                                ", [
                                    'option_id' => $option_id,
                                    'language_id' => $language_id,
                                    'name' => $name
                                ]);
                            }

                            $cache_product_option[$name] = [
                                'option_id' => $option_id,
                                'type' => 'select',
                                'values' => []
                            ];
                        }

                        if (!$this->getValue($cache_product_option, [$name, 'values', $value])) {
                            $option_value_id = $this->dev5->db->query("
                                INSERT INTO
                                    `" . DB_PREFIX . "option_value`
                                SET
                                    `option_id` = :option_id,
                                    `sort_order` = 0,
                                    `image` = ''
                            ", [
                                'option_id' => $option_id
                            ])->lastId();
                            foreach ($this->getLanguages() as $language_id) {
                                $this->dev5->db->query("
                                    INSERT INTO
                                        `" . DB_PREFIX . "option_value_description`
                                    SET
                                        `option_value_id` = :option_value_id,
                                        `language_id` = :language_id,
                                        `option_id` = :option_id,
                                        `name` = :name
                                ", [
                                    'option_value_id' => $option_value_id,
                                    'language_id' => $language_id,
                                    'option_id' => $option_id,
                                    'name' => $value
                                ]);
                            }

                            $cache_product_option[$name]['values'][$value] = $option_value_id;
                        }

                        $current_option = [
                            'product_option_id' => '',
                            'product_option_value_id' => '',
                            'quantity' => 0,
                            'price' => $price,
                            'price_prefix' => $price_prefix,
                            'points' => 0,
                            'points_prefix' => '+',
                            'weight' => 0,
                            'weight_prefix' => '+',
                        ];
                    }

                    if (!isset($product_options[$name])) {
                        $product_options[$name] = [
                            'product_option_id' => $this->getValue($current_option, 'product_option_id'),
                            'product_option_value' => [],
                            'option_id' => $this->getValue($cache_product_option, [$name, 'option_id']),
                            'name' => $name,
                            'type' => $this->getValue($cache_product_option, [$name, 'type'], 'select'),
                            'value' => '',
                            'required' => 0
                        ];
                    }

                    if (!isset($product_options[$name]['product_option_value'][$value])) {
                        $product_options[$name]['required']++;
                        $product_options[$name]['product_option_value'][$value] = [
                            'product_option_value_id' => $this->getValue($current_option, 'product_option_value_id'),
                            'option_value_id' => $this->getValue($cache_product_option, [$name, 'values', $value]),
                            'sku' => $this->getValue($variacao, 'codigo'),
                            'quantity' => $this->getValue($current_option, 'quantity', 0),
                            'subtract' => $this->getValue($current_option, 'subtract', 1),
                            'price' => $this->getValue($current_option, 'price', 0),
                            'price_prefix' => $this->getValue($current_option, 'price_prefix', '+'),
                            'points' => $this->getValue($current_option, 'points', 0, 'floatval'),
                            'points_prefix' => $this->getValue($current_option, 'points_prefix', '+'),
                            'weight' => $this->getValue($current_option, 'weight', 0, 'floatval'),
                            'weight_prefix' => $this->getValue($current_option, 'weight_prefix', '+')
                        ];
                    }

                    $product_options[$name]['product_option_value'][$value]['quantity'] = max(
                        $product_options[$name]['product_option_value'][$value]['quantity'],
                        $this->getValue($variacao, ['estoque', 'saldoVirtualTotal'], 0)
                    );

                    $product_options[$name]['required']++;
                }

                $quantity += $this->getValue($variacao, ['estoque', 'saldoVirtualTotal'], 0);
            }

            $product_options = array_values($product_options);

            foreach ($product_options as &$product_option) {
                $product_option['required'] = $product_option['required'] >= $count;
                $product_option['product_option_value'] = array_values($product_option['product_option_value']);
            }

            $data['quantity'] = $quantity;
            $data['product_option'] = $product_options;
        }
    }

    /**
     * @param array $data
     * @param array $bling_produto
     * @return array
     */
    private function importProductOptionsPrices($data, $bling_produto)
    {
        $combinations = [];
        $options = [];
        $matriz = [];
        $prices = [];

        foreach ($bling_produto['variacoes'] as $variacao) {
            $names = explode(';', $this->getValue($variacao, ['variacao', 'nome']) ?: []);
            $combinations[] = array_merge([$this->getValue($variacao, 'preco', 0)], $names);
            $options = array_unique(array_merge($options, $names));
        }

        foreach ($combinations as $combination) {
            $price = $combination[0] - $data['price'];
            $calc = [];

            foreach ($options as $option) {
                $calc[] = in_array($option, $combination) ? 1 : 0;
            }

            $matriz[] = $calc;
            $prices[] = $price;
        }

        $countMemory = count($matriz);
        $countOptions = count($options);
        for ($i = 0; $i < $countOptions; $i++) {
            $max = $i;
            for ($j = $i + 1; $j < $countMemory; $j++) {
                if (abs($matriz[$j][$i]) > abs($matriz[$max][$i])) {
                    $max = $j;
                }
            }

            if ($max !== $i) {
                $temp = $matriz[$i];
                $matriz[$i] = $matriz[$max];
                $matriz[$max] = $temp;

                $temp = $prices[$i];
                $prices[$i] = $prices[$max];
                $prices[$max] = $temp;
            }
            for ($j = $i + 1; $j < $countMemory; $j++) {
                if ($matriz[$i][$i] === 0) {
                    continue;
                }

                $factor = $matriz[$j][$i] / $matriz[$i][$i];
                for ($k = $i; $k < $countOptions; $k++) {
                    $matriz[$j][$k] -= $factor * $matriz[$i][$k];
                }
                $prices[$j] -= $factor * $prices[$i];
            }
        }

        $result = array_fill(0, $countOptions, 0);
        for ($i = min($countMemory, $countOptions) - 1; $i >= 0; $i--) {
            $sum = 0;
            for ($j = $i + 1; $j < $countOptions; $j++) {
                $sum += $matriz[$i][$j] * $result[$j];
            }

            if ($matriz[$i][$i] != 0) {
                $result[$i] = ($prices[$i] - $sum) / $matriz[$i][$i];
            }
        }

        return array_combine($options, $result);
    }

    /**
     * @param array $bling_product
     * @return array
     */
    private function importProductCategoryAutocomplete($bling_product, $payload, $prefix = '')
    {
        $result = ['value' => '', 'text' => ''];

        if ($bling_category_id = $this->getValue($bling_product, ['categoria', 'id'])) {
            if ($category_id = CategoryEntity::getInstance()->importCategory($bling_category_id, $payload, $prefix)) {
                $result['value'] = $category_id;
                $result['text'] = CategoryEntity::getInstance()->getCategory($category_id)['full_name'];
            }
        }

        return $result;
    }

    /**
     * @param array $bling_product
     * @return array
     */
    private function importProductGroupProductAutocomplete($bling_product)
    {
        $result = ['value' => '', 'text' => ''];

        if ($bling_grupo_produto_id = $this->getValue($bling_product, ['tributacao', 'grupoProduto', 'id'], '')) {
            static $bling_grupo_produto = null;
            if ($bling_grupo_produto === null) {
                $bling_grupo_produto = Api::getInstance()->getGruposProdutos(false, ['cache' => 7 * 24 * 60 * 60]);
                $bling_grupo_produto = @array_column($bling_grupo_produto ?: [], 'nome', 'id');
            }

            if (isset($bling_grupo_produto[$bling_grupo_produto_id])) {
                $result = [
                    'value' => $bling_grupo_produto_id,
                    'text' => $bling_grupo_produto[$bling_grupo_produto_id]
                ];
            }
        }

        return $result;
    }

    /**
     * @param int $product_id
     * @param array $variacao
     * @return array
     */
    private function importProductVariationToProductOptions($product_id, $variacao)
    {
        if (!($options = $this->getValue($variacao, ['variacao', 'nome']))) {
            return [];
        }

        $language_id = (int)$this->dev5->field('language_id');

        $product_options = [];

        $options = array_map(static function ($i) {
            return explode(':', $i);
        }, explode(';', $options));

        foreach ($options as $option) {
            $product_options[] = $this->dev5->db->query("
                SELECT
                    pov.product_option_value_id
                FROM
                    `" . DB_PREFIX . "product_option_value` pov
                INNER JOIN
                    `" . DB_PREFIX . "option_description` od
                    ON pov.option_id = od.option_id
                    AND od.language_id = :language_id
                INNER JOIN
                    `" . DB_PREFIX . "option_value_description` ovd
                    ON pov.option_value_id = ovd.option_value_id
                    AND ovd.language_id = :language_id
                WHERE
                    pov.product_id = :product_id
                    AND od.name = :name
                    AND ovd.name = :value
            ", [
                'language_id' => $language_id,
                'product_id' => $product_id,
                'name' => $option[0],
                'value' => $option[1]
            ])->column();
        }

        sort($product_options);

        return $product_options;
    }

    /**
     * @param string $bling_id
     * @param self $entity
     * @return bool
     */
    private function importProductStock($bling_id, $prefix = '', &$error = null)
    {
        $this->dev5->debug("$prefix • IMPORT-PRODUCT-STOCK", "$bling_id");

        if (!$bling_id) {
            $error = ['message' => 'FAIL, NO-PRODUCT-ID', 'file' => __FILE__ . ':' . __LINE__];
            $this->dev5->debug("$prefix |→ FAIL", "NO-PRODUCT-ID");
            return false;
        }

        $bling_product = Api::getInstance()->getProduto($bling_id);
        if (!empty($bling_product['variacao']['produtoPai']['id'])) {
            $this->dev5->debug("$prefix | • IMPORT-PRODUCT-STOCK-OPTION-PARENT", $bling_product['variacao']['produtoPai']['id']); // phpcs:ignore
            if (!$this->importProductStock($bling_product['variacao']['produtoPai']['id'], "$prefix |", $error)) { // phpcs:ignore
                $this->dev5->debug("$prefix | |→ FAIL");
                $this->dev5->debug("$prefix |→ FAIL");
                return false;
            }

            $this->dev5->debug("$prefix | |→ SUCCESS");
            $this->dev5->debug("$prefix |→ SUCCESS");
            return true;
        }

        if (!$bling_product) {
            $error = ['message' => 'FAIL, NO-DATA', 'file' => __FILE__ . ':' . __LINE__, 'debug' => Api::getInstance()->last()]; // phpcs:ignore
            $this->dev5->debug("$prefix |→ FAIL", "NO-DATA");
            return false;
        }

        $entity = self::findOne(['bling_id' => $bling_id]) ?: new self();
        $entity->bling_id = $bling_id;
        $entity->deposito = $entity->deposito ?: $this->dev5->field('deposito');

        if (!$entity->product_id && ($current = $this->getProductByCode($bling_product['codigo']))) {
            $entity->product_id = $current['product_id'];
        }

        if (!$entity->product_id) {
            $error = ['message' => "FAIL, NO-FOUND-PRODUCT[$bling_id]", 'file' => __FILE__ . ':' . __LINE__]; // phpcs:ignore
            $this->dev5->debug("$prefix |→ FAIL", "NO-FOUND-PRODUCT[$bling_id]");
            return false;
        }

        $product_quantity = $this->getValue($bling_product, ['estoque', 'saldoVirtualTotal'], 0);
        $product_options = [];

        if (!empty($bling_product['variacoes'])) {
            foreach ($bling_product['variacoes'] as $variacao) {
                $variacao_quantity = $this->getValue($variacao, ['estoque', 'saldoVirtualTotal'], 0);

                foreach ($this->importProductVariationToProductOptions($entity->product_id, $variacao) as $product_option_value_id) { // phpcs:ignore
                    !isset($product_options[$product_option_value_id]) && $product_options[$product_option_value_id] = 0; // phpcs:ignore
                    $product_options[$product_option_value_id] = max($product_options[$product_option_value_id], $variacao_quantity); // phpcs:ignore
                }
            }

            $product_quantity = max($product_quantity, array_sum($product_options));
        }

        $current_quantity = $this->dev5->db->query("
            SELECT
                p.quantity
            FROM
                `" . DB_PREFIX . "product` p
            WHERE
                p.product_id = :product_id
        ", [
            'product_id' => $entity->product_id
        ])->column();

        $this->dev5->debug("$prefix | • $entity->product_id: $current_quantity → $product_quantity");
        if ($current_quantity == $product_quantity) {
            $this->dev5->debug("$prefix | |→ SKIP", "SAME QUANTITY");
        } else {
            $this->dev5->debug("$prefix | | • UPDATE");
            $this->dev5->db->queryUpdate(DB_PREFIX . "product", [
                'values' => ['quantity' => $product_quantity],
                'where' => ['product_id' => $entity->product_id]
            ]);
            $this->dev5->debug("$prefix | | |→ SUCCESS");
        }
        $this->dev5->debug("$prefix | |→ SUCCESS");

        if ($product_options) {
            foreach ($product_options as $product_option_value_id => $product_option_quantity) {
                $current_option_quantity = $this->dev5->db->query("
                    SELECT
                        pov.quantity
                    FROM
                        `" . DB_PREFIX . "product_option_value` pov
                    WHERE
                        pov.product_option_value_id = :product_option_value_id
                ", [
                    'product_option_value_id' => $product_option_value_id
                ])->column();

                $this->dev5->debug("$prefix | • $entity->product_id-$product_option_value_id: $current_option_quantity → $product_option_quantity"); // phpcs:ignore

                if ($current_option_quantity == $product_option_quantity) {
                    $this->dev5->debug("$prefix | |→ SKIP", "SAME QUANTITY");
                } else {
                    $this->dev5->debug("$prefix | | • UPDATE");
                    $this->dev5->db->queryUpdate(DB_PREFIX . "product_option_value", [
                        'values' => ['quantity' => $product_option_quantity],
                        'where' => ['product_option_value_id' => $product_option_value_id]
                    ]);
                }
            }
        }

        $this->dev5->debug("$prefix |→ SUCCESS");
        return true;
    }

    /**
     * @param EIEntity\Payload $payload
     * @param string $prefix
     * @return bool
     */
    public function importProducts($payload, $prefix = '')
    {
        $this->dev5->debug("$prefix • IMPORT-PRODUCTS", $payload->cursor);

        $filter = ['criterio' => 5, 'tipo' => 'P'];

        if (!empty($payload->input['bling_id']) || !empty($payload->input['name'])) {
            if (!empty($payload->input['bling_id'])) {
                $bling_produto_ids = $payload->input['bling_id'];
            }

            if (!empty($payload->input['name'])) {
                $temp_ids = array_column($payload->input['name'], 'value');
                $bling_produto_ids = isset($bling_produto_ids)
                    ? array_intersect($bling_produto_ids, $temp_ids)
                    : $temp_ids;
            }

            $filter['idsProdutos'] = empty($bling_produto_ids) ? [-1] : array_unique($bling_produto_ids);
        }

        !empty($payload->input['codigos'])
        && $filter['codigos'] = $payload->input['codigo'];

        !empty($payload->input['status'])
        && $filter['criterio'] = $payload->input['status'];

        !empty($payload->input['date_added']['start'])
        && $filter['dataInclusaoInicial'] = "{$payload->input['date_added']['start']} 00:00:00";

        !empty($payload->input['date_added']['end'])
        && $filter['dataInclusaoFinal'] = "{$payload->input['date_added']['end']} 23:59:59";

        !empty($payload->input['date_modified']['start'])
        && $filter['dataAlteracaoInicial'] = "{$payload->input['date_modified']['start']} 00:00:00";

        !empty($payload->input['date_modified']['end'])
        && $filter['dataAlteracaoFinal'] = "{$payload->input['date_modified']['end']} 23:59:59";

        $limit = 100;

        $page = ['start' => $payload->cursor + 1, 'limit' => $limit];

        $rows = Api::getInstance()->getProdutos($filter, ['page' => $page]);

        foreach ($rows as $row) {
            EIEntity::fromArray([
                'id' => $row['id'],
                'label' => "#$row[id] $row[nome]"
            ])->save(['returning' => false]);
            $payload->total++;

            $this->dev5->debug("$prefix | • $row[id] → PENDING");
        }

        $count = count($rows);

        $this->dev5->debug("$prefix |→ SUCCESS", "$count/$limit");

        return $count < $limit;
    }

    /**
     * @param EIEntity\Payload $payload
     * @param string $prefix
     * @return bool
     */
    public function importProductsApprove($payload, $prefix = '')
    {
        $this->dev5->debug("$prefix • IMPORT-PRODUCTS-APPROVE", $payload->cursor);

        if ($entity = EIEntity::findOne(['status' => EIEntity::PENDING])) {
            $this->dev5->debug("$prefix | • IMPORT-PRODUCT-APPROVE", $entity->id);
            QueueEntity::publish("import_product:$entity->id", ['event' => 'ei'], QueueEntity::PROCESSING, $queue);

            $entity->status = $this->importProduct($entity->id, $payload, "$prefix | |", $error) ? EIEntity::SUCCESS : EIEntity::FAIL; // phpcs:ignore
            $entity->save(['returning' => false]);

            ($queue->status = ($entity->status === EIEntity::SUCCESS ? QueueEntity::SUCCESS : QueueEntity::FAIL))
            === QueueEntity::FAIL && $queue->payload = array_merge($queue->payload, [
                'error' => !empty($error) ? $error : Api::getInstance()->error(),
                'debug' => Api::getInstance()->last()
            ]);
            $queue->save(['returning' => false]);

            $payload->total++;
            $this->dev5->debug("$prefix | |→ " . ($entity->status === EIEntity::SUCCESS ? 'SUCCESS' : 'FAIL'));
        }

        $finished = !$entity || EIEntity::findTotal(['status' => EIEntity::PENDING]) < 1;

        $this->dev5->debug("$prefix |→ " . ($finished ? 'FINISHED' : 'NEXT'));

        return $finished;
    }

    /**
     * @param array $data
     * @param string $prefix
     * @return bool
     */
    public function importCallback($data, $prefix = '')
    {
        $this->dev5->debug("$prefix • IMPORT-PRODUCT-CALLBACK", $data['id']);
        QueueEntity::publish("callback_product_stock:$data[id]", ['event' => 'callback'], QueueEntity::PROCESSING, $queue); // phpcs:ignore
        $queue->status = QueueEntity::FAIL;

        try {
            if ($this->importProductStock($data['id'], "$prefix |", $error)) {
                $this->dev5->debug("$prefix |→ SUCCESS");
                $queue->status = QueueEntity::SUCCESS;
                return true;
            }
            $this->dev5->debug("$prefix |→ FAIL");
        } catch (\Exception $e) {
            $error = "$e";
            $this->dev5->debug("$prefix |→ FAIL", "$e");
        } catch (\Throwable $e) {
            $error = "$e";
            $this->dev5->debug("$prefix |→ FAIL", "$e");
        } finally {
            $queue->status === QueueEntity::FAIL && $queue->payload = array_merge($queue->payload, [
                'error' => !empty($error) ? $error : Api::getInstance()->error(),
                'debug' => Api::getInstance()->last()
            ]);
            $queue->save(['returning' => false]);
        }
        return false;
    }
}
