<?php

/**
 * This file is part of Dev5 Integração ERP Bling API V3 for OpenCart
 *
 * @author    Dev5™ <developer@dev5.com.br>
 * @copyright Dev5™ 2025. Todos os direitos reservados.
 * @license   Distribuição/reprodução não autorizada estará sujeita às penas das leis 9609/98 e 9610/98
 * @link      https://dev5.com.br/integracao-erp-bling-api-v3
 * @version   4.0.6
 */

namespace Dev5\Library\Bling;

use Dev5\Contract\Database\Condition as C;
use Exception;
use Throwable;

use function Dev5\Opencart\ocver;

/** @Dev5\Orm\Entity(['name' => DB_PREFIX . 'd5bling3_order', 'auto_migrate' => false]) */
final class OrderEntity extends BaseEntity
{
    /**
     * @var int
     * @Dev5\Orm\Field(['type' => 'int', 'primary' => true])
     */
    public $order_id;

    /**
     * @var string|null
     * @Dev5\Orm\Field(['type' => 'varchar', 'length' => 64, 'nullable' => true])
     */
    public $bling_id;

    /**
     * @var string|null
     * @Dev5\Orm\Field(['type' => 'varchar', 'length' => 64, 'nullable' => true])
     */
    public $loja_id;

    /**
     * @var string|null
     * @Dev5\Orm\Field(['type' => 'varchar', 'length' => 64, 'nullable' => true])
     */
    public $numero_loja;

    /**
     * @param int $order_id
     * @return array|null
     */
    public function getOrder($order_id)
    {
        $data = $this->dev5->db->query("
             SELECT
                o.*
            FROM
                " . DB_PREFIX . "order o
            WHERE
                o.order_id = :order_id
        ", [
            'order_id' => $order_id
        ])->row();

        if (!$data) {
            return null;
        }

        $data['custom_field'] = ocver('2.1', '<')
            ? unserialize($data['custom_field'])
            : json_decode($data['custom_field'], true);

        $data['payment_custom_field'] = ocver('2.1', '<')
            ? unserialize($data['payment_custom_field'])
            : json_decode($data['payment_custom_field'], true);

        $data['shipping_custom_field'] = ocver('2.1', '<')
            ? unserialize($data['shipping_custom_field'])
            : json_decode($data['shipping_custom_field'], true);

        $data['totals'] = array_map(static function ($order_total) {
            $order_total['title'] = html_entity_decode($order_total['title']);
            $order_total['value'] = (float)$order_total['value'];
            $order_total['sort_order'] = (int)$order_total['sort_order'];
            return $order_total;
        }, $this->dev5->db->query("
            SELECT
                ot.code,
                ot.title,
                ot.value,
                ot.sort_order
            FROM
                " . DB_PREFIX . "order_total ot
            WHERE
                ot.order_id = :order_id
        ", [
            'order_id' => $order_id
        ])->all());

        $data['products'] = array_map(function ($product) {
            $product['product_id'] = (int)$product['product_id'];
            $product['order_product_id'] = (int)$product['order_product_id'];
            $product['name'] = html_entity_decode($product['name']);
            $product['model'] = html_entity_decode($product['model']);
            $product['quantity'] = (int)$product['quantity'];
            $product['price'] = (float)$product['price'];
            $product['total'] = (float)$product['total'];
            $product['tax'] = (float)$product['tax'];
            $product['reward'] = (int)$product['reward'];
            $product['weight'] = (float)$product['weight'];
            $product['weight_class_id'] = (int)$product['weight_class_id'];
            $product['option'] = $this->dev5->db->query("
                SELECT
                    oo.product_option_id,
                    oo.product_option_value_id,
                    {$this->getColumnOptionSKU()} AS sku,
                    oo.name,
                    oo.value,
                    pov.weight,
                    pov.weight_prefix,
                    o.type
                FROM
                    " . DB_PREFIX . "order_option oo
                LEFT JOIN
                    " . DB_PREFIX . "product_option_value pov
                    ON pov.product_option_value_id = oo.product_option_value_id
                LEFT JOIN
                    " . DB_PREFIX . "option o
                    ON o.option_id = pov.option_id
                WHERE
                    oo.order_product_id = :order_product_id
            ", [
                'order_product_id' => $product['order_product_id']
            ])->all();
            unset($product['order_product_id']);
            return $product;
        }, $this->dev5->db->query("
            SELECT
                op.order_product_id,
                op.product_id,
                op.name,
                op.model,
                p.sku,
                op.quantity,
                op.price,
                op.total,
                op.tax,
                op.reward,
                p.weight,
                p.weight_class_id
            FROM
                " . DB_PREFIX . "order_product op
            LEFT JOIN
                " . DB_PREFIX . "product p
                ON p.product_id = op.product_id
            WHERE
                op.order_id = :order_id
        ", [
            'order_id' => $order_id
        ])->all());

        $data['history'] = array_map(static function ($order_total) {
            $order_total['comment'] = html_entity_decode($order_total['comment']);
            return $order_total;
        }, $this->dev5->db->query("
            SELECT
                oh.order_history_id,
                oh.order_status_id,
                oh.comment
            FROM
                " . DB_PREFIX . "order_history oh
            WHERE
                oh.order_id = :order_id
            ORDER BY
                oh.order_history_id
        ", [
            'order_id' => $order_id
        ])->all());

        return $data;
    }

    /**
     * @param array $order
     * @param string $field
     * @return mixed
     */
    private function getOrderValue($order, $field)
    {
        if (stripos($field, 'custom_field_') === 0) {
            return $this->getValue($order, ['custom_field', substr($field, 13)]);
        }

        if (stripos($field, 'payment_custom_field_') === 0) {
            return $this->getValue($order, ['payment_custom_field', substr($field, 21)]);
        }

        if (stripos($field, 'shipping_custom_field_') === 0) {
            return $this->getValue($order, ['shipping_custom_field', substr($field, 22)]);
        }

        switch ($field) {
            case 'customer':
                return "{$this->getValue($order, 'firstname')} {$this->getValue($order, 'lastname')}";

            case 'payment_customer':
                return "{$this->getValue($order, 'payment_firstname')} {$this->getValue($order, 'payment_lastname')}";

            case 'shipping_customer':
                return "{$this->getValue($order, 'shipping_firstname')} {$this->getValue($order, 'shipping_lastname')}";

            case 'payment_zone_code':
                return @$this->model_localisation_zone->getZone((int)$order['payment_zone_id'])['code'] ?: null;

            case 'shipping_zone_code':
                return @$this->model_localisation_zone->getZone((int)$order['shipping_zone_id'])['code'] ?: null;

            default:
                return $this->getValue($order, $field);
        }
    }

    /**
     * @param array $order
     * @param string $field
     * @param mixed $value
     * @return void
     */
    private function setOrderValue(&$order, $field, $value)
    {
        if (!$field) {
            return;
        }

        if (str_starts_with($field, 'custom_field_')) {
            $order['custom_field'][substr($field, 13)] = $value;
            return;
        }

        if (str_starts_with($field, 'payment_custom_field_')) {
            $order['payment_custom_field'][substr($field, 21)] = $value;
            return;
        }

        if (str_starts_with($field, 'shipping_custom_field_')) {
            $order['shipping_custom_field'][substr($field, 22)] = $value;
            return;
        }

        switch ($field) {
            case 'customer':
            case 'payment_customer':
            case 'shipping_customer':
                $prefix = str_replace('customer', '', $field);
                $value = explode(' ', $value, 2);
                $order["{$prefix}firstname"] = $value[0];
                $order["{$prefix}lastname"] = isset($value[1]) ? $value[1] : '';
                return;

            case 'payment_zone':
            case 'payment_zone_id':
            case 'payment_zone_code':
            case 'shipping_zone_id':
            case 'shipping_zone_code':
                $prefix = str_replace(['zone_id', 'zone_code'], '', $field);

                $zone = current($this->dev5->db->querySelect(DB_PREFIX . 'zone', [
                    'where' => [str_contains($field, 'zone_id') ? 'zone_id' : 'code' => $value]
                ]));

                $order["{$prefix}zone_id"] = isset($zone['zone_id']) ? $zone['zone_id'] : 0;
                $order["{$prefix}zone"] = isset($zone['name']) ? $zone['name'] : '';
                return;

            case 'payment_country':
            case 'shipping_country':
                $prefix = str_replace('country', '', $field);

                $country = current($this->dev5->db->querySelect(DB_PREFIX . 'country', [
                    'where' => ['name' => $value]
                ])) ?: ['country_id' => 30, 'name' => 'Brasil'];

                $order["{$prefix}country_id"] = isset($country['country_id']) ? $country['country_id'] : 0;
                $order["{$prefix}country"] = isset($country['name']) ? $country['name'] : '';
                return;

            default:
                $order[$field] = $value;
                break;
        }
    }

    /**
     * @param array $order
     * @param string $prefix
     * @return void
     */
    private function setCustomerSeller($order, $prefix = '')
    {
        $this->dev5->debug("$prefix • SET-CUSTOMER-SELLER", $order['order_id']);

        switch (@$this->dev5->field('ei')['update_seller_on_event']) {
            case 'last':
                $seller = @$this->dev5->db->querySelect(DB_PREFIX . 'order', [
                    'columns' => ['custom_field'],
                    'where' => [
                        'customer_id' => $order['customer_id'],
                        'order_status_id' => array_merge(
                            $this->config->get('config_processing_status') ?: [],
                            $this->config->get('config_complete_status') ?: []
                        ),
                        'custom_field' => ['operator' => 'LIKE', 'value' => '%"d5bling_seller_%"%']
                    ],
                    'order_by' => ['order_id' => 'DESC'],
                    'limit' => 1
                ])[0]['custom_field'];

                is_string($seller) && $seller = ocver('2.1', '<')
                    ? unserialize($seller)
                    : json_decode($seller, true);

                $seller = $this->getValue($seller ?: [], 'customer_d5bling_seller') ?: null;
                $this->dev5->debug("$prefix | • LAST MODE");
                break;

            case 'current':
                $seller = $this->getValue($order, ['custom_field', 'customer_d5bling_seller']);
                $this->dev5->debug("$prefix | • CURRENT MODE");
                break;

            default:
                $this->dev5->debug("$prefix |→ SKIP, NO-UPDATE");
                return;
        }

        $customer_custom_fields = @$this->dev5->db->querySelect(DB_PREFIX . 'customer', [
            'columns' => ['custom_field'],
            'where' => ['customer_id' => $order['customer_id']]
        ])[0]['custom_field'] ?: [];

        is_string($customer_custom_fields) && $customer_custom_fields = ocver('2.1', '<')
            ? unserialize($customer_custom_fields)
            : json_decode($customer_custom_fields, true);

        !$customer_custom_fields && $customer_custom_fields = [];

        if (!$seller) {
            unset($customer_custom_fields['customer_d5bling_seller']);
            $this->dev5->debug("$prefix | |→ UNSET");
        } else {
            $customer_custom_fields['customer_d5bling_seller'] = $seller;
            $this->dev5->debug("$prefix | |→ SET, $seller");
        }

        $customer_custom_fields = ocver('2.1', '<')
            ? serialize($customer_custom_fields)
            : json_encode($customer_custom_fields);

        $this->dev5->db->queryUpdate(DB_PREFIX . 'customer', [
            'values' => ['custom_field' => $customer_custom_fields],
            'where' => ['customer_id' => $order['customer_id']]
        ]);

        $this->dev5->debug("$prefix |→ SUCCESS");
    }

    /**
     * @param mixed $value
     * @param string $mask
     * @return string
     */
    private function getMaskedValue($value, $mask)
    {
        $value = preg_replace('/\D/', '', $value);

        switch ($mask) {
            case 'tel':
                return preg_replace('/^(\d{2})(\d{4,5})(\d{4})$/', '($1) $2-$3', $value);

            case 'document':
                return preg_replace(
                    strlen($value) > 11
                        ? '/^(\d{2})(\d{3})(\d{3})(\d{4})(\d{2})$/'
                        : '/^(\d{3})(\d{3})(\d{3})(\d{2})$/',
                    strlen($value) > 11
                        ? '$1.$2.$3/$4-$5'
                        : '$1.$2.$3-$4',
                    $value
                );

            case 'postcode':
                return preg_replace('/^(\d{5})(\d{3})$/', '$1-$2', $value);
        }

        return $value;
    }

    /**
     * @param array $payload
     * @param string $prefix
     * @return string|false
     */
    public function exportOrder($payload, $prefix = '')
    {
        $this->dev5->debug("$prefix • EXPORT-ORDER", $this->order_id);

        $ignore_id = $this->dev5->field('ei')['export_order_ignore_id'];
        if ($ignore_id && $ignore_id > $this->order_id) {
            $this->dev5->debug("$prefix |→ SKIP", "IGNORED-ID");
            return true;
        }

        $order = @$this->getOrder($this->order_id);
        if (!$order) {
            $this->dev5->debug("$prefix |→ FAIL", "NO-DATA");
            return false;
        }

        $ignore_date = $this->dev5->field('ei')['export_order_ignore_date'];
        if ($ignore_date && strtotime($ignore_date) > strtotime($order['date_added'])) {
            $this->dev5->debug("$prefix |→ SKIP", "IGNORED-DATE");
            return true;
        }

        if (!$this->getValue($order, 'order_status_id')) {
            $this->dev5->debug("$prefix |→ SKIP", "NO-STATUS");
            return true;
        }

        $this->setCustomerSeller($order, "$prefix |");

        $b2o = $this->dev5->field('order_status');
        $o2b = array_column($b2o, 'status_id');

        if (!in_array($order['order_status_id'], $o2b)) {
            $this->dev5->debug("$prefix |→ SKIP", "IGNORED-STATUS[$order[order_status_id]]");
            return true;
        }

        if (!($bling_contato_id = $this->exportOrderContact($order, "$prefix |"))) {
            $this->dev5->debug("$prefix |→ FAIL", "NO-CONTACT");
            return false;
        }

        if (!($items = $this->exportOrderNormalizeItems($order, $payload, $prefix))) {
            $this->dev5->debug("$prefix |→ FAIL", "NO-ITEMS");
            return false;
        }

        $this->numero_loja = $this->numero_loja ?: $this->order_id;
        $this->loja_id = $this->loja_id ?: $this->dev5->field('channel_id');
        $bling_pedido = null;

        if ($this->bling_id && !($bling_pedido = Api::getInstance()->getPedidoVenda($this->bling_id))) {
            $this->bling_id = null;
        }

        if (!$this->bling_id) {
            $temp = @Api::getInstance()->getPedidosVenda(['numerosLojas' => [$this->numero_loja], 'idLoja' => $this->loja_id])[0]['id']; // phpcs:ignore
            !empty($temp) && $bling_pedido = Api::getInstance()->getPedidoVenda($this->bling_id = $temp);
            $bling_pedido && $this->bling_id = $bling_pedido['id'];
        }

        $this->exportOrderNormalizeTotals($order);

        // phpcs:disable
        $data = $this->filterNullable([
            'numero' => $this->getValue($bling_pedido, 'numero'),
            'numeroLoja' => $this->numero_loja,
            'data' => datez('Y-m-d', $this->getValue($order, 'date_added')),
            'dataSaida' => $this->getValue($bling_pedido, 'dataSaida'),
            'dataPrevista' => $this->getValue($bling_pedido, 'dataPrevista'),
            'contato' => ['id' => $bling_contato_id],
            'loja' => ['id' => $this->loja_id],
            'numeroPedidoCompra' => $this->getValue($bling_pedido, 'numeroPedidoCompra'),
            'outrasDespesas' => $this->getValue($bling_pedido, 'outrasDespesas'),
            'observacoes' => $this->getValue($order, 'comment'),
            'observacoesInternas' => $this->getValue($bling_pedido, 'observacoesInternas'),
            'desconto' => [
                'valor' => $this->currency->convert($this->getValue($order, ['totals', 'discount']) ?: 0, $order['currency_code'], $this->dev5->field('currency_code')),
                'unidade' => 'REAL'
            ],
            'categoria' => $this->filterNullable([
                'id' => $this->getValue($bling_pedido, ['categoria', 'id'])
            ]),
            'tributacao' => $this->filterNullable([
                'totalICMS' => $this->getValue($bling_pedido, 'totalICMS'),
                'totalIPI' => $this->getValue($bling_pedido, 'totalIPI'),
            ]),
            'itens' => $items,
            'parcelas' => $this->exportOrderNormalizeInstallments($order),
            'transporte' => $this->exportOrderNormalizeTransport($order, $bling_pedido),
            'vendedor' => $this->filterNullable([
                'id' => substr($this->getOrderValue($order, 'custom_field_customer_d5bling_seller') ?: '', strlen('d5bling_seller_'))
            ]),
            'intermediador' => $this->filterNullable([
                'cnpj' => $this->getValue($bling_pedido, ['intermediador', 'cnpj']),
                'nomeUsuario' => $this->getValue($bling_pedido, ['intermediador', 'nomeUsuario'])
            ]),
            'taxas' => $this->filterNullable([
                'taxaComissao' => $this->getValue($bling_pedido, ['taxas', 'taxaComissao']) ?: null,
                'custoFrete' => $this->getValue($bling_pedido, ['taxas', 'custoFrete']) ?: null,
                'valorBase' => $this->getValue($bling_pedido, ['taxas', 'valorBase']) ?: null
            ])
        ]);
        // phpcs:enable

        if (!$this->bling_id) {
            $this->dev5->debug("$prefix | • CREATE");
            $this->bling_id = Api::getInstance()->postPedidoVenda($data);
        } elseif (array_multidiff($data, $bling_pedido)) {
            $this->dev5->debug("$prefix | • UPDATE");
            $this->bling_id = Api::getInstance()->putPedidoVenda($bling_pedido['id'], $data);
        } else {
            $this->dev5->debug("$prefix | • SKIP, SAME-DATA");
            $this->bling_id = $bling_pedido['id'];
        }

        if (!$this->bling_id) {
            $this->dev5->debug("$prefix | |→ FAIL", Api::getInstance()->error());
            $this->dev5->debug("$prefix |→ FAIL");
            return false;
        }

        !$bling_pedido && $bling_pedido = Api::getInstance()->getPedidoVenda($this->bling_id);
        if (!$bling_pedido) {
            $this->dev5->debug("$prefix | |→ FAIL", Api::getInstance()->error());
            $this->dev5->debug("$prefix |→ FAIL", "BLING-NOT-FOUND");
            return false;
        }

        $this->save(['returning' => false]);
        $this->dev5->debug("$prefix | |→ SUCCESS");

        if (!($bling_status_id = $this->exportOrderStatus($order, $bling_pedido, $b2o, "$prefix |"))) {
            $this->dev5->debug("$prefix |→ FAIL");
            return false;
        }

        if (!$this->exportOrderStock($order, $bling_pedido, $b2o, $bling_status_id, "$prefix |")) {
            $this->dev5->debug("$prefix |→ FAIL");
            return false;
        }

        $this->dev5->debug("$prefix |→ SUCCESS, $this->bling_id");
        return $this->bling_id;
    }

    /**
     * @param array $order
     * @param string $prefix
     * @return string
     */
    private function exportOrderContact($order, $prefix = '')
    {
        $this->dev5->debug("$prefix • EXPORT-ORDER-CONTACT", $order['order_id']);

        $contato = $this->dev5->field('contato');

        $document = preg_replace(
            '/\D/',
            '',
            $this->getOrderValue($order, $contato['cnpj']) ?: $this->getOrderValue($order, $contato['cpf'])
        );

        $bling_contato = @Api::getInstance()->getContatos(['numeroDocumento' => $document, 'criterio' => 1]);
        !empty($bling_contato[0]['id']) && $bling_contato = Api::getInstance()->getContato($bling_contato[0]['id']);

        $data = $this->filterNullable([
            'nome' => $this->getOrderValue($order, $contato['nome']),
            'codigo' => $this->getOrderValue($order, $contato['codigo']),
            'situacao' => 'A',
            'numeroDocumento' => $document,
            'telefone' => $this->getOrderValue($order, $contato['telefone']),
            'celular' => $this->getOrderValue($order, $contato['celular']),
            'fantasia' => $this->getOrderValue($order, $contato['fantasia']),
            'tipo' => strlen($document) > 11 ? 'J' : 'F',
            'indicadorIe' => $this->getOrderValue($order, $contato['indicadorIe']) ?: $contato['defaultIndicadorIe'],
            'ie' => $this->getOrderValue($order, $contato['ie']),
            'rg' => $this->getOrderValue($order, $contato['rg']),
            'inscricaoMunicipal' => $this->getOrderValue($order, $contato['im']),
            'orgaoEmissor' => $this->getOrderValue($order, $contato['orgaoEmissor']),
            'email' => $this->getOrderValue($order, $contato['email']),
            'endereco' => $this->filterNullable([
                'geral' => $this->filterNullable([
                    'endereco' => $this->getOrderValue($order, $contato['endereco']['geral']['endereco']),
                    'cep' => preg_replace('/\D/', '', $this->getOrderValue($order, $contato['endereco']['geral']['cep'])), // phpcs:ignore
                    'bairro' => $this->getOrderValue($order, $contato['endereco']['geral']['bairro']),
                    'municipio' => trim(preg_replace('/\s+/', ' ', $this->getOrderValue($order, $contato['endereco']['geral']['municipio']))), // phpcs:ignore
                    'uf' => $this->getOrderValue($order, $contato['endereco']['geral']['uf']),
                    'numero' => $this->getOrderValue($order, $contato['endereco']['geral']['numero']),
                    'complemento' => $this->getOrderValue($order, $contato['endereco']['geral']['complemento'])
                ]),
                'cobranca' => $this->filterNullable([
                    'endereco' => $this->getOrderValue($order, $contato['endereco']['cobranca']['endereco']),
                    'cep' => preg_replace('/\D/', '', $this->getOrderValue($order, $contato['endereco']['cobranca']['cep'])), // phpcs:ignore
                    'bairro' => $this->getOrderValue($order, $contato['endereco']['cobranca']['bairro']),
                    'municipio' => trim(preg_replace('/\s+/', ' ', $this->getOrderValue($order, $contato['endereco']['cobranca']['municipio']))), // phpcs:ignore
                    'uf' => $this->getOrderValue($order, $contato['endereco']['cobranca']['uf']),
                    'numero' => $this->getOrderValue($order, $contato['endereco']['cobranca']['numero']),
                    'complemento' => $this->getOrderValue($order, $contato['endereco']['cobranca']['complemento'])
                ])
            ]),
            'vendedor' => $this->filterNullable([
                'id' => substr($this->getOrderValue($order, 'custom_field_customer_d5bling_seller') ?: '', strlen('d5bling_seller_')) // phpcs:ignore
            ]),
            'dadosAdicionais' => $this->filterNullable([
                'dataNascimento' => $this->getOrderValue($order, $contato['dataNascimento']),
                'sexo' => $this->getOrderValue($order, $contato['sexo']),
                'naturalidade' => $this->getOrderValue($order, $contato['naturalidade']),
            ]),
            'financeiro' => $this->getValue($bling_contato, 'financeiro'),
            'tiposContato' => $this->getValue($bling_contato, 'tiposContato')
        ]);

        if (!$bling_contato) {
            $this->dev5->debug("$prefix | • CREATE");
            $bling_contato_id = Api::getInstance()->postContato($data);
        } elseif (array_multidiff($data, $bling_contato)) {
            $this->dev5->debug("$prefix | • UPDATE");
            $bling_contato_id = Api::getInstance()->putContato($bling_contato['id'], $data);
        } else {
            $this->dev5->debug("$prefix | • SKIP, SAME-DATA");
            $bling_contato_id = $bling_contato['id'];
        }

        if (!$bling_contato_id) {
            $this->dev5->debug("$prefix | |→ FAIL", Api::getInstance()->error());
            $this->dev5->debug("$prefix |→ FAIL");
            return false;
        }

        $this->dev5->debug("$prefix | |→ SUCCESS");
        $this->dev5->debug("$prefix |→ SUCCESS", $bling_contato_id);
        return $bling_contato_id;
    }

    /**
     * @param array $order
     * @param array $payload
     * @param string $prefix
     * @return array|false
     */
    private function exportOrderNormalizeItems($order, $payload, $prefix = '')
    {
        $products = [];

        foreach ($order['products'] as $product) {
            $entity = ProductEntity::getInstance($product['product_id']);

            $codigo = $entity->makeCode($product['product_id'], $product['sku'], array_filter(array_map(static function ($option) { // phpcs:ignore
                return !empty($option['product_option_value_id'])
                    ? ['id' => $option['product_option_value_id'], 'sku' => isset($option['sku']) ? $option['sku'] : null] // phpcs:ignore
                    : false;
            }, $product['option'] ?: [])) ?: null);

            if (!($product_data = $entity->getCacheBlingProduct($codigo, 'codigo'))) {
                $p = $payload;
                $p['force'] = true;
                $p['filter_by_codigo'] = $codigo;
                if (!$entity->exportProduct($p, "$prefix |")) {
                    return false;
                }

                if (!($product_data = $entity->getCacheBlingProduct($codigo, 'codigo'))) {
                    return false;
                }
            }

            $products[] = $this->filterNullable([
                'codigo' => $codigo,
                'unidade' => $entity->unidade,
                'quantidade' => $product['quantity'],
                'desconto' => 0,
                'valor' => $this->currency->convert($product['price'], $order['currency_code'], $this->dev5->field('currency_code')), // phpcs:ignore
                //'aliquotaIPI' => ?,
                'descricao' => $product_data['name'],
                //'descricaoDetalhada' => $product['name'],
                'produto' => ['id' => $product_data['bling_id']],
                //'comissao' => [
                //    'base' => ?,
                //    'aliquota' => ?,
                //    'valor' => ?
                //]
            ]);

            // Force recheck after order export
            QueueEntity::publish("export_product:$product[product_id]", ['event' => "order", "force" => true]);
        }

        return $products;
    }

    /**
     * @param array $order
     * @return void
     */
    private function exportOrderNormalizeTotals(&$order)
    {
        $shipping = 0;
        $sub_total = 0;

        foreach ($order['totals'] as $total) {
            switch ($total['code']) {
                case 'shipping':
                    $shipping = $total['value'];
                    break;
                case 'sub_total':
                    $sub_total = $total['value'];
                    break;
            }
        }

        $total = $order['total'];

        $order['totals'] = [
            'shipping' => $this->currency->convert($shipping, $order['currency_code'], $this->dev5->field('currency_code')), // phpcs:ignore
            'discount' => $this->currency->convert($total - $sub_total - $shipping, $order['currency_code'], $this->dev5->field('currency_code')) * -1, // phpcs:ignore
            'total' => $this->currency->convert($total, $order['currency_code'], $this->dev5->field('currency_code')), // phpcs:ignore
        ];
    }

    /**
     * @param array $order
     * @return array|null
     */
    private function exportOrderNormalizeInstallments($order)
    {
        if (ocver('4.0.2', '>=')) {
            $order['payment_method'] = json_decode($order['payment_method'], true);
            $order['payment_code'] = $order['payment_method']['code'];
            $order['payment_method'] = $order['payment_method']['name'];
        }

        $methods = $this->dev5->field('payment_method') ?: [];

        $bling_pagamento_id = array_find_key($methods, static function ($i) use ($order) {
            return $i['text'] && $i['value'] && $i['value'] === $order['payment_code'];
        });

        if (!$bling_pagamento_id) {
            return null;
        }

        $bling_pagamento = $methods[$bling_pagamento_id];

        $installment_num = 1;
        if (!empty($bling_pagamento['regex'])) {
            foreach ($order['history'] as $history) {
                if (preg_match('/(?:\s|^)(\d{1,2})x(?:\s|$)/', $history['comment'], $matches)) {
                    $installment_num = (int)$matches[1];
                    break;
                }
            }
        }

        $installment_inc = intval(isset($bling_pagamento['days']) ? $bling_pagamento['days'] : 0);
        $installment_date = date_create($order['date_added']);
        $installment_total = round($order['total'] / $installment_num, 3);

        $installments = [];
        for ($i = 0; $i < $installment_num; $i++) {
            $installment_date = $installment_date->modify("+$installment_inc days");

            $installments[] = [
                'dataVencimento' => $installment_date->format("Y-m-d"),
                'valor' => $installment_total,
                'formaPagamento' => ['id' => $bling_pagamento_id]
            ];
        }

        return $installments;
    }

    /**
     * @param array $order
     * @param array $bling_pedido
     * @return array
     */
    private function exportOrderNormalizeTransport($order, $bling_pedido)
    {
        if (ocver('4.0.2', '>=')) {
            $order['shipping_method'] = json_decode($order['shipping_method'], true);
            $order['shipping_code'] = $order['shipping_method']['code'];
            $order['shipping_method'] = $order['shipping_method']['name'];
        }

        $methods = $this->dev5->field('shipping_method') ?: [];

        $bling_logistica_id = array_find_key($methods, static function ($i) use ($order) {
            return $i['text'] && $i['value'] && preg_match("/$i[value]/", $order['shipping_code']);
        });

        $bling_logistica = $bling_logistica_id ? $methods[$bling_logistica_id] : null;

        $pesoBruto = 0;
        foreach ($order['products'] as $product) {
            $produtoPesoBruto = $product['weight'] * $product['quantity'];
            if ($product['option']) {
                foreach ($product['option'] as $option) {
                    if ($option['weight_prefix'] === '+') {
                        $produtoPesoBruto += $option['weight'];
                    } else {
                        $produtoPesoBruto -= $option['weight'];
                    }
                }
            }
            $pesoBruto += $this->weight->convert($produtoPesoBruto, $product['weight_class_id'], $this->dev5->field('weight_class_id')); // phpcs:ignore
        }

        $contato = $this->dev5->field('contato');

        $volumes = $this->getValue($bling_pedido, ['transporte', 'volumes']) ?: [[]];
        if ($serivce = $this->getValue($bling_logistica, 'service') ?: $this->getValue($bling_logistica, 'logistic')) {
            $volumes[0]['servico'] = $serivce;
        }

        return $this->filterNullable([
            'fretePorConta' => $this->getValue($bling_logistica, 'fretePorConta'),
            'frete' => $this->currency->convert($this->getValue($order, ['totals', 'shipping']) ?: 0, $order['currency_code'], $this->dev5->field('currency_code')), // phpcs:ignore
            'quantidadeVolumes' => $this->getValue($bling_pedido, ['transporte', 'quantidadeVolumes']) ?: 1,
            'pesoBruto' => $pesoBruto,
            'prazoEntrega' => $this->getValue($bling_pedido, ['transporte', 'prazoEntrega']) ?: 0,
            'contato' => $this->filterNullable([
                'id' => $this->getValue($bling_pedido, ['transporte', 'contato', 'id']),
                'nome' => $this->getValue($bling_logistica, 'logistic')
            ]),
            'etiqueta' => $this->filterNullable([
                'nome' => $this->getOrderValue($order, $contato['endereco']['geral']['nome']),
                'endereco' => $this->getOrderValue($order, $contato['endereco']['geral']['endereco']),
                'cep' => preg_replace('/\D/', '', $this->getOrderValue($order, $contato['endereco']['geral']['cep'])), // phpcs:ignore
                'bairro' => $this->getOrderValue($order, $contato['endereco']['geral']['bairro']),
                'municipio' => trim(preg_replace('/\s+/', ' ', $this->getOrderValue($order, $contato['endereco']['geral']['municipio']))), // phpcs:ignore
                'uf' => $this->getOrderValue($order, $contato['endereco']['geral']['uf']),
                'numero' => $this->getOrderValue($order, $contato['endereco']['geral']['numero']),
                'complemento' => $this->getOrderValue($order, $contato['endereco']['geral']['complemento'])
            ]),
            'volumes' => $volumes
        ]);
    }

    /**
     * @param array $order
     * @param array $bling_pedido
     * @param array $b2o
     * @param string $prefix
     * @return bool
     */
    private function exportOrderStatus($order, $bling_pedido, $b2o, $prefix = '')
    {
        $this->dev5->debug("$prefix • EXPORT-ORDER-STATUS", $order['order_id']);

        $bling_status_id = array_find_key($b2o, static function ($status) use ($order) {
            return $status['status_id'] == $order['order_status_id'];
        });

        if (!$bling_status_id) {
            $this->dev5->debug("$prefix |→ SKIP", "IGNORED-STATUS[$order[order_status_id]]");
            return true;
        }

        if ($bling_pedido['situacao']['id'] == $bling_status_id) {
            $this->dev5->debug("$prefix |→ SKIP, SAME-STATUS");
        } elseif (!Api::getInstance()->patchPedidoVendaSituacao($bling_pedido['id'], $bling_status_id)) {
            $this->dev5->debug("$prefix |→ FAIL", Api::getInstance()->error());
            return false;
        } else {
            $this->dev5->debug("$prefix |→ SUCCESS", $bling_status_id);
        }

        return $bling_status_id;
    }

    /**
     * @param array $order
     * @param array $bling_pedido
     * @param array $b2o
     * @param string $bling_status_id
     * @param string $prefix
     * @return bool
     */
    private function exportOrderStock($order, $bling_pedido, $b2o, $bling_status_id, $prefix = '')
    {
        $this->dev5->debug("$prefix • EXPORT-ORDER-STOCK", $order['order_id']);

        if (empty($b2o[$bling_status_id]['stock'])) {
            $this->dev5->debug("$prefix |→ SKIP, IGNORED-STOCK");
            return true;
        }

        if ($b2o[$bling_status_id]['stock'] === '1') {
            $this->dev5->debug("$prefix | • REDUCE");
            $result = Api::getInstance()->pedidoPedidoVendaLancarEstoque($bling_pedido['id']);
        } else {
            $this->dev5->debug("$prefix | • REVERSE");
            $result = Api::getInstance()->pedidoPedidoVendaEstornarEstoque($bling_pedido['id']);
        }

        if (!$result) {
            $this->dev5->debug("$prefix | |→ FAIL", Api::getInstance()->error());
            $this->dev5->debug("$prefix |→ FAIL");
            return false;
        }

        $this->dev5->debug("$prefix | |→ SUCCESS");
        $this->dev5->debug("$prefix |→ SUCCESS");
        return true;
    }

    /**
     * @param array $payload
     * @param string $prefix
     * @return bool
     */
    public function exportDeleteOrder($payload, $prefix = '')
    {
        $this->dev5->debug("$prefix • DELETE-ORDER", $this->order_id);

        if (!$this->bling_id) {
            $this->dev5->debug("$prefix |→ SUCCESS", "NO-DATA");
            return true;
        }

        if (!Api::getInstance()->deletePedidoVenda($this->bling_id)) {
            $this->dev5->debug("$prefix |→ FAIL", Api::getInstance()->error());
            return false;
        }

        OrderEntity::delete($this->order_id);

        $this->dev5->debug("$prefix |→ SUCCESS");
        return true;
    }

    /**
     * @param EIEntity\Payload $payload
     * @param string $prefix
     * @return bool
     */
    public function exportOrders($payload, $prefix = '')
    {
        $this->dev5->debug("$prefix • EXPORT-PRODUCTS", $payload->cursor);

        $where = [];
        $values = [];

        if (!empty($payload->input['id']) || !empty($payload->input['name'])) {
            if (!empty($payload->input['id'])) {
                $order_ids = $payload->input['id'];
            }

            if (!empty($payload->input['name'])) {
                $temp_ids = array_column($payload->input['name'], 'value');
                $order_ids = isset($order_ids)
                    ? array_intersect($order_ids, $temp_ids)
                    : $temp_ids;
            }

            $where[] = "o.order_id IN :order_id";
            $values['order_id'] = empty($order_ids) ? [-1] : array_unique($order_ids);
        }

        if (!empty($payload->input['status'])) {
            $where[] = "o.order_status_id IN :status";
            $values['status'] = $payload->input['status'] ?: [-1];
        } else {
            $where[] = "o.order_status_id > 0";
        }

        if (!empty($payload->input['date_added']['start']) && !empty($payload->input['date_added']['end'])) {
            $where[] = "o.date_added BETWEEN :date_added_start AND :date_added_end";
            $values['date_added_start'] = "{$payload->input['date_added']['start']} 00:00:00";
            $values['date_added_end'] = "{$payload->input['date_added']['end']} 23:59:59";
        } elseif (!empty($payload->input['date_added']['start'])) {
            $where[] = "o.date_added >= :date_added_start";
            $values['date_added_start'] = "{$payload->input['date_added']['start']} 00:00:00";
        } elseif (!empty($payload->input['date_added']['end'])) {
            $where[] = "o.date_added <= :date_added_end";
            $values['date_added_end'] = "{$payload->input['date_added']['end']} 23:59:59";
        }

        if (!empty($payload->input['date_modified']['start']) && !empty($payload->input['date_modified']['end'])) {
            $where[] = "o.date_modified BETWEEN :date_modified_start AND :date_modified_end";
            $values['date_modified_start'] = "{$payload->input['date_modified']['start']} 00:00:00";
            $values['date_modified_end'] = "{$payload->input['date_modified']['end']} 23:59:59";
        } elseif (!empty($payload->input['date_modified']['start'])) {
            $where[] = "o.date_modified >= :date_modified_start";
            $values['date_modified_start'] = "{$payload->input['date_modified']['start']} 00:00:00";
        } elseif (!empty($payload->input['date_modified']['end'])) {
            $where[] = "o.date_modified <= :date_modified_end";
            $values['date_modified_end'] = "{$payload->input['date_modified']['end']} 23:59:59";
        }

        $where = empty($where) ? '' : 'WHERE ' . implode(' AND ', $where);
        $limit = 100;
        $offset = $payload->cursor * $limit;

        $rows = $this->dev5->db->query("
            SELECT
                o.order_id,
                CONCAT(o.firstname, ' ', o.lastname) AS `customer`,
                o.total,
                o.currency_code
            FROM
                " . DB_PREFIX . "order o
            $where
            ORDER BY o.order_id
            LIMIT $offset, $limit;
        ", $values)->all();

        foreach ($rows as $row) {
            $total = $this->currency->convert($row['total'], $row['currency_code'], $this->config->get('config_currency')); // phpcs:ignore
            $total = $this->currency->format($total, $this->config->get('config_currency'));

            EIEntity::fromArray([
                'id' => $row['order_id'],
                'label' => "#$row[order_id] $row[customer] $total"
            ])->save(['returning' => false]);
            $payload->total++;

            $this->dev5->debug("$prefix | • $row[order_id] → PENDING");
        }

        $count = count($rows);

        $this->dev5->debug("$prefix |→ SUCCESS", "$count/$limit");

        return $count < $limit;
    }

    /**
     * @param EIEntity\Payload $payload
     * @param string $prefix
     * @return bool
     */
    public function exportOrdersApprove($payload, $prefix = '')
    {
        $this->dev5->debug("$prefix • EXPORT-ORDERS-APPROVE", $payload->cursor);

        $limit = 100;
        $entities = EIEntity::findMany(['status' => EIEntity::PENDING], ['limit' => $limit]);

        foreach ($entities as $entity) {
            $this->dev5->debug("$prefix | • EXPORT-ORDERS-APPROVE", $entity->id);

            QueueEntity::publish("export_order:$entity->id", ['event' => 'ei']);
            $payload->total++;

            $entity->status = EIEntity::QUEUE;
            $entity->save(['returning' => false]);

            $this->dev5->debug("$prefix | |→ QUEUE");
        }

        $count = count($entities);

        $this->dev5->debug("$prefix |→ SUCCESS", "$count/$limit");

        return $count < $limit;
    }

    /**
     * @param string $bling_id
     * @param string $prefix
     * @return int|false
     */
    private function importOrder($bling_id, $prefix = '', &$error = null)
    {
        $this->dev5->debug("$prefix • IMPORT-ORDER", "$bling_id");

        $bling_order = Api::getInstance()->getPedidoVenda($bling_id);
        if (!$bling_order) {
            $error = ['message' => 'FAIL, NO-DATA', 'file' => __FILE__ . ':' . __LINE__, 'debug' => Api::getInstance()->last()]; // phpcs:ignore
            $this->dev5->debug("$prefix |→ FAIL", "NO-DATA", Api::getInstance()->error());
            return false;
        }

        $entity = self::findOne(['bling_id' => $bling_id]) ?: new self();
        $entity->bling_id = $bling_id;

        if ($entity->order_id && (!($current = $this->getOrder($entity->order_id)))) {
            $entity->order_id = 0;
        }

        if (!$entity->order_id && $this->getValue($bling_order, ['loja', 'id']) == $this->dev5->field('channel_id')) {
            $entity->order_id = (int)$this->getValue($bling_order, 'numeroLoja');
        }

        // phpcs:disable
        $current = empty($current) ? [] : $current;

        $data = $current;

        $b2o = $this->dev5->field('order_status');
        $bling_status_id = $this->getValue($bling_order, ['situacao', 'id']);
        if (empty($b2o[$bling_status_id]['status_id'])) {
            $error = ['message' => "SKIP, IGNORED-STATUS[$bling_status_id]", 'file' => __FILE__ . ':' . __LINE__];
            $this->dev5->debug("$prefix |→ SKIP", "IGNORED-STATUS[$bling_status_id]");
            return false;
        }

        $language = $this->model_localisation_language->getLanguage((int)$this->dev5->field('language_id'));
        $currency = $this->model_localisation_currency->getCurrencyByCode($this->dev5->field('currency_code'));

        $data['subscription_id'] = $this->getValue($data, 'subscription_id', '');
        $data['invoice_prefix'] = $this->getValue($data, 'invoice_prefix', $this->config->get('config_invoice_prefix'));
        $data['store_id'] = $this->getValue($data, 'store_id', $this->config->get('config_store_id'));
        $data['store_name'] = $this->getValue($data, 'store_name', "Integração Bling #$bling_id");
        $data['store_url'] = $this->getValue($data, 'store_url', "https://www.bling.com.br/vendas.php#edit/$bling_id");
        $this->importOrderNormalizeCustomer($data, $bling_order);
        $this->importOrderNormalizePayment($data, $bling_order);
        $this->importOrderNormalizeShipping($data, $bling_order);
        $data['comment'] = $this->getValue($bling_order, 'observacoes');
        $data['affiliate_id'] = $this->getValue($data, 'affiliate_id', '');
        $data['commission'] = $this->getValue($data, 'commission', '');
        $data['marketing_id'] = $this->getValue($data, 'marketing_id', '');
        $data['tracking'] = $this->getValue($data, 'tracking', '');
        $data['language_id'] = $language['language_id'];
        $data['language_code'] = $language['code']; // 4x
        $data['currency_id'] = $currency['currency_id'];
        $data['currency_code'] = $currency['code'];
        $data['currency_value'] = $currency['value'];
        $data['ip'] = $this->getValue($data, 'ip', '');
        $data['forwarded_ip'] = $this->getValue($data, 'forwarded_ip', '');
        $data['user_agent'] = $this->getValue($data, 'user_agent', 'Integração ERP Bling API V3');
        $data['accept_language'] = $this->getValue($data, 'accept_language', '');
        $data['date_added'] = $this->getValue($bling_order, 'data') ?: $this->getValue($data, 'date_added') ?: date('Y-m-d H:i:s');
        if (!$this->importOrderNormalizeProduct($data, $bling_order, "$prefix |")) {
            $this->dev5->debug("$prefix |→ FAIL", "PRODUCTS");
            return false;
        }
        $this->importOrderNormalizeTotals($data, $bling_order);

        $entity->numero_loja = $this->getValue($bling_order, 'numeroLoja');
        $entity->loja_id = $this->getValue($bling_order, ['loja', 'id']);
        // phpcs:enable

        try {
            $compare = $data;
            unset($compare['marketing_id']);
            unset($compare['tracking']);
            unset($compare['vouchers']);
            if (ocver('4', '<')) {
                unset($compare['subscription_id']);
                unset($compare['payment_address_id']);
                unset($compare['shipping_address_id']);
                unset($compare['language_code']);
                $compare['products'] = array_map(static function ($product) {
                    unset($product['master_id']);
                    unset($product['subscription']);
                    return $product;
                }, $compare['products']);
                $compare['totals'] = array_map(static function ($total) {
                    unset($total['extension']);
                    return $total;
                }, $compare['totals']);
            }

            $order_id = $entity->order_id;
            if ($order_id && !$this->dev5->db->query("SELECT COUNT(order_id) FROM " . DB_PREFIX . "order WHERE order_id = :order_id", ['order_id' => $order_id])->column()) { // phpcs:ignore
                $order_id = null;
            }

            self::ignore(true);
            if (!array_multidiff($compare, $current)) {
                $this->dev5->debug("$prefix | • SKIP", "SAME DATA");
            } elseif (!$order_id) {
                $this->dev5->debug("$prefix | • CREATE");
                $data['order_status_id'] = $bling_status_id;
                $order_id = $this->importOrderSave(null, $data, "$prefix | |");
            } else {
                $this->dev5->debug("$prefix | • UPDATE");
                $order_id = $this->importOrderSave($order_id, $data, "$prefix | |");
            }
        } catch (Exception $e) {
            $error = "$e";
            $this->dev5->debug("$prefix | |→ FAIL", "$e");
            $this->dev5->debug("$prefix |→ FAIL");
            return false;
        } catch (Throwable $e) {
            $error = "$e";
            $this->dev5->debug("$prefix | |→ FAIL", "$e");
            $this->dev5->debug("$prefix |→ FAIL");
            return false;
        } finally {
            self::ignore(false);
        }

        if (!$order_id) {
            $error = ['message' => 'FAIL, NO-ORDER-ID', 'file' => __FILE__ . ':' . __LINE__];
            $this->dev5->debug("$prefix | |→ FAIL", "NO-ORDER-ID");
            $this->dev5->debug("$prefix |→ FAIL");
            return false;
        }

        $this->dev5->debug("$prefix | |→ SUCCESS");
        $entity->order_id = $order_id;
        $entity->save(['returning' => false]);

        if (!$this->importOrderStatus($entity->order_id, $bling_status_id, "$prefix |", $error)) {
            $this->dev5->debug("$prefix |→ FAIL");
            return false;
        }

        $this->dev5->debug("$prefix |→ SUCCESS", $entity->order_id);
        return $entity->order_id;
    }

    /**
     * @param array $data
     * @param array $bling_order
     */
    private function importOrderNormalizeCustomer(&$data, $bling_order)
    {
        // phpcs:disable
        $contato = $this->dev5->field('contato');

        $bling_contato = Api::getInstance()->getContato($this->getValue($bling_order, ['contato', 'id']));

        $customer = $this->dev5->db->query("
            SELECT
                c.customer_id AS customer_id,
                c.firstname AS firstname,
                c.lastname AS lastname,
                c.email AS email,
                c.telephone AS telephone,
                c.custom_field AS custom_field,
                a.address_id AS address_id,
                a.firstname AS address_firstname,
                a.lastname AS address_lastname,
                a.company AS address_company,
                a.address_1 AS address_address_1,
                a.address_2 AS address_address_2,
                a.city AS address_city,
                a.postcode AS address_postcode,
                a.country_id AS address_country_id,
                c2.name AS address_country,
                a.zone_id AS address_zone_id,
                z.name AS address_zone,
                a.custom_field AS address_custom_field
            FROM
                " . DB_PREFIX . "customer c
            LEFT JOIN
                " . DB_PREFIX . "address a
                ON " . (ocver('4', '>=')
                ? 'c.customer_id = a.customer_id AND a.default'
                : 'c.address_id = a.address_id'
            ) . "
            LEFT JOIN
                " . DB_PREFIX . "country c2
                ON c2.country_id = a.country_id
            LEFT JOIN
                " . DB_PREFIX . "zone z
                ON z.zone_id = a.zone_id
            WHERE
                email = :email
        ", [
            'email' => $this->getValue($bling_contato, 'email')
        ])->row();

        $fields = [
            'customer_id' => $this->getValue($customer, 'customer_id', 0),
            'customer_group_id' => $this->getValue($customer, 'customer_group_id', $this->config->get('config_customer_group_id')),
            'firstname' => $this->getValue($customer, 'firstname'),
            'lastname' => $this->getValue($customer, 'lastname'),
            'email' => $this->getValue($customer, 'email'),
            'telephone' => $this->getValue($customer, 'telephone'),
            'custom_field' => $this->getValue($customer, 'custom_field', ''),
            'payment_address_id' => $this->getValue($customer, 'address_id', 0), // 4x
            'payment_firstname' => $this->getValue($customer, 'address_firstname', ''),
            'payment_lastname' => $this->getValue($customer, 'address_lastname', ''),
            'payment_company' => $this->getValue($customer, 'address_company', ''),
            'payment_address_1' => $this->getValue($customer, 'address_address_1', ''),
            'payment_address_2' => $this->getValue($customer, 'address_address_2', ''),
            'payment_city' => $this->getValue($customer, 'address_city', ''),
            'payment_postcode' => $this->getValue($customer, 'address_postcode', ''),
            'payment_country' => $this->getValue($customer, 'address_country', ''),
            'payment_country_id' => $this->getValue($customer, 'address_country_id', 0),
            'payment_zone' => $this->getValue($customer, 'address_zone', ''),
            'payment_zone_id' => $this->getValue($customer, 'address_zone_id', 0),
            'payment_address_format' => $this->getValue($customer, 'address_format', ''),
            'payment_custom_field' => $this->getValue($customer, 'address_custom_field', ''),
            'shipping_address_id' => $this->getValue($customer, 'address_id', 0), // 4x
            'shipping_firstname' => $this->getValue($customer, 'address_firstname', ''),
            'shipping_lastname' => $this->getValue($customer, 'address_lastname', ''),
            'shipping_company' => $this->getValue($customer, 'address_company', ''),
            'shipping_address_1' => $this->getValue($customer, 'address_address_1', ''),
            'shipping_address_2' => $this->getValue($customer, 'address_address_2', ''),
            'shipping_city' => $this->getValue($customer, 'address_city', ''),
            'shipping_postcode' => $this->getValue($customer, 'address_postcode', ''),
            'shipping_country' => $this->getValue($customer, 'address_country', ''),
            'shipping_country_id' => $this->getValue($customer, 'address_country_id', 0),
            'shipping_zone' => $this->getValue($customer, 'address_zone', ''),
            'shipping_zone_id' => $this->getValue($customer, 'address_zone_id', 0),
            'shipping_address_format' => $this->getValue($customer, 'address_format', ''),
            'shipping_custom_field' => $this->getValue($customer, 'address_custom_field', '')
        ];
        foreach ($fields as $key => $value) {
            !array_key_exists($key, $data) && $data[$key] = $value;
        }

        $data['custom_field'] && !is_array($data['custom_field']) && $data['custom_field'] = ocver('2.1', '<')
            ? unserialize($data['custom_field'])
            : json_decode($data['custom_field'], true);
        !$data['custom_field'] && $data['custom_field'] = [];

        $data['payment_custom_field'] && !is_array($data['payment_custom_field']) && $data['payment_custom_field'] = ocver('2.1', '<')
            ? unserialize($data['payment_custom_field'])
            : json_decode($data['payment_custom_field'], true);
        !$data['payment_custom_field'] && $data['payment_custom_field'] = [];

        $data['shipping_custom_field'] && !is_array($data['shipping_custom_field']) && $data['shipping_custom_field'] = ocver('2.1', '<')
            ? unserialize($data['shipping_custom_field'])
            : json_decode($data['shipping_custom_field'], true);
        !$data['shipping_custom_field'] && $data['shipping_custom_field'] = [];

        $this->setOrderValue($data, $contato['nome'], $this->getValue($bling_order, ['contato', 'nome']));
        $this->setOrderValue($data, $contato['fantasia'], $this->getValue($bling_contato, 'fantasia'));
        $this->setOrderValue($data, $contato['email'], $this->getValue($bling_contato, 'email'));
        $this->setOrderValue($data, $contato['telefone'], $this->getMaskedValue($this->getValue($bling_contato, 'telefone'), 'tel'));
        $this->setOrderValue($data, $contato['celular'], $this->getMaskedValue($this->getValue($bling_contato, 'celular'), 'tel'));
        if (strlen($document = preg_replace('/\D/', '', $this->getValue($bling_order, ['contato', 'numeroDocumento']))) === 14) {
            $this->setOrderValue($data, $contato['cpf'], null);
            $this->setOrderValue($data, $contato['cnpj'], $this->getMaskedValue($document, 'document'));
        } else {
            $this->setOrderValue($data, $contato['cnpj'], null);
            $this->setOrderValue($data, $contato['cpf'], $this->getMaskedValue($document, 'document'));
        }
        $this->setOrderValue($data, $contato['rg'], preg_replace('/\D/', '', $this->getValue($bling_contato, 'rg')));
        $this->setOrderValue($data, $contato['orgaoEmissor'], preg_replace('/\D/', '', $this->getValue($bling_contato, 'orgaoEmissor')));
        $this->setOrderValue($data, $contato['ie'], preg_replace('/\D/', '', $this->getValue($bling_contato, 'ie')));
        $this->setOrderValue($data, $contato['indicadorIe'], $this->getValue($bling_contato, 'indicadorIe'));
        $this->setOrderValue($data, $contato['im'], preg_replace('/\D/', '', $this->getValue($bling_contato, 'inscricaoMunicipal')));
        $this->setOrderValue($data, $contato['dataNascimento'], $this->getValue($bling_contato, ['dadosAdicionais', 'dataNascimento']));
        $this->setOrderValue($data, $contato['sexo'], $this->getValue($bling_contato, ['dadosAdicionais', 'sexo']));
        $this->setOrderValue($data, $contato['naturalidade'], $this->getValue($bling_contato, ['dadosAdicionais', 'naturalidade']));
        $this->setOrderValue($data, 'custom_field_customer_d5bling_seller', ($i = $this->getValue($bling_order, ['vendedor', 'id'])) ? "d5bling_seller_$i" : '');

        $this->setOrderValue($data, $contato['endereco']['cobranca']['nome'], $this->getValue($bling_contato, ['nome']));
        $this->setOrderValue($data, $contato['endereco']['cobranca']['cep'], $this->getMaskedValue($this->getValue($bling_contato, ['endereco', 'cobranca', 'cep']), 'postcode'));
        $this->setOrderValue($data, $contato['endereco']['cobranca']['endereco'], $this->getValue($bling_contato, ['endereco', 'cobranca', 'endereco']));
        $this->setOrderValue($data, $contato['endereco']['cobranca']['numero'], $this->getValue($bling_contato, ['endereco', 'cobranca', 'numero']));
        $this->setOrderValue($data, $contato['endereco']['cobranca']['bairro'], $this->getValue($bling_contato, ['endereco', 'cobranca', 'bairro']));
        $this->setOrderValue($data, $contato['endereco']['cobranca']['municipio'], $this->getValue($bling_contato, ['endereco', 'cobranca', 'municipio']));
        $this->setOrderValue($data, $contato['endereco']['cobranca']['uf'], $this->getValue($bling_contato, ['endereco', 'cobranca', 'uf']));
        $this->setOrderValue($data, $contato['endereco']['cobranca']['complemento'], $this->getValue($bling_contato, ['endereco', 'cobranca', 'complemento']));
        $this->setOrderValue($data, 'payment_country', $this->getValue($bling_order, ['pais', 'nome']));

        $this->setOrderValue($data, $contato['endereco']['geral']['nome'], $this->getValue($bling_order, ['etiqueta', 'nome']) ?: $this->getValue($bling_contato, ['nome']));
        $this->setOrderValue($data, $contato['endereco']['geral']['cep'], $this->getMaskedValue($this->getValue($bling_order, ['etiqueta', 'cep']) ?: $this->getValue($bling_contato, ['endereco', 'geral', 'cep']), 'postcode'));
        $this->setOrderValue($data, $contato['endereco']['geral']['endereco'], $this->getValue($bling_order, ['etiqueta', 'endereco']) ?: $this->getValue($bling_contato, ['endereco', 'geral', 'endereco']));
        $this->setOrderValue($data, $contato['endereco']['geral']['numero'], $this->getValue($bling_order, ['etiqueta', 'numero']) ?: $this->getValue($bling_contato, ['endereco', 'geral', 'numero']));
        $this->setOrderValue($data, $contato['endereco']['geral']['bairro'], $this->getValue($bling_order, ['etiqueta', 'bairro']) ?: $this->getValue($bling_contato, ['endereco', 'geral', 'bairro']));
        $this->setOrderValue($data, $contato['endereco']['geral']['municipio'], $this->getValue($bling_order, ['etiqueta', 'municipio']) ?: $this->getValue($bling_contato, ['endereco', 'geral', 'municipio']));
        $this->setOrderValue($data, $contato['endereco']['geral']['uf'], $this->getValue($bling_order, ['etiqueta', 'uf']) ?: $this->getValue($bling_contato, ['endereco', 'geral', 'uf']));
        $this->setOrderValue($data, $contato['endereco']['geral']['complemento'], $this->getValue($bling_order, ['etiqueta', 'complemento']) ?: $this->getValue($bling_contato, ['endereco', 'geral', 'complemento']));
        $this->setOrderValue($data, 'shipping_country', $this->getValue($bling_order, ['etiqueta', 'nomePais']) ?: $this->getValue($bling_order, ['pais', 'nome']));
        // phpcs:enable
    }

    /**
     * @param array $data
     * @param array $bling_order
     */
    private function importOrderNormalizePayment(&$data, $bling_order)
    {
        $bling_pagamento_id = $this->getValue($bling_order, ['parcelas', 0, 'formaPagamento', 'id']);

        $methods = $this->dev5->field('payment_method');

        $method = isset($methods[$bling_pagamento_id]) ? $methods[$bling_pagamento_id] : [];

        if (empty($method['value']) || empty($method['text'])) {
            $payment_method = 'Não especificado';
            $payment_code = '';
        } else {
            $payment_code = $methods[$bling_pagamento_id]['value'];
            $payment_method = trim(preg_replace("/\(.*\)$/", '', $methods[$bling_pagamento_id]['text']));
        }

        if (ocver('4.0.2', '>=')) {
            $data['payment_method'] = [
                'name' => $payment_method,
                'code' => $payment_code
            ];
        } else {
            $data['payment_method'] = $payment_method;
            $data['payment_code'] = $payment_code;
        }
    }

    /**
     * @param array $data
     * @param array $bling_order
     */
    private function importOrderNormalizeShipping(&$data, $bling_order)
    {
        $logistic = $this->getValue($bling_order, ['transporte', 'contato', 'nome']);
        $service = $this->getValue($bling_order, ['transporte', 'volumes', '0', 'servico']);

        if ($logistic) {
            $methods = $this->dev5->field('shipping_method');

            $method = array_find($methods, static function ($method) use ($logistic, $service) {
                return $method['logistic'] == $logistic && $method['service'] == $service;
            });
        }

        if (empty($method['value']) || empty($method['text'])) {
            $shipping_code = 'd5bling';
            $shipping_method = $logistic . ($service ? " ($service)" : '');
        } else {
            $shipping_code = $method['value'];
            $shipping_method = trim(preg_replace("/\(.*\)$/", '', $method['text']));
        }

        if (str_ends_with($shipping_code, '.*')) {
            $shipping_code = rtrim($shipping_code, '.*');
            $shipping_code = "$shipping_code.$shipping_code";
        }

        if (ocver('4.0.2', '>=')) {
            $data['shipping_method'] = [
                'name' => $shipping_method,
                'code' => $shipping_code
            ];
        } else {
            $data['shipping_method'] = $shipping_method;
            $data['shipping_code'] = $shipping_code;
        }
    }

    /**
     * @param array $data
     * @param array $bling_order
     * @param string $prefix
     * @return bool
     */
    private function importOrderNormalizeProduct(&$data, $bling_order, $prefix = '')
    {
        $this->dev5->debug("$prefix • IMPORT-ORDER-PRODUCTS");

        $data['products'] = [];

        foreach ($this->getValue($bling_order, 'itens', []) as $item) {
            $bling_produto_id = $this->getValue($item, ['produto', 'id']);
            $bling_produto_codigo = $this->getValue($item, ['codigo']);

            $this->dev5->debug("$prefix | • ORDER-PRODUCT", $bling_produto_id);

            $entity = ProductEntity::findOne(C::_OR(C::eq('bling_id', $bling_produto_id), C::eq('codigo', $bling_produto_codigo))); // phpcs:ignore
            if (!$entity || !$entity->product_id) {
                if (!ProductEntity::getInstance()->importProduct($bling_produto_id, null, "$prefix | |")) {
                    $this->dev5->debug("$prefix | |→ FAIL");
                    $this->dev5->debug("$prefix |→ FAIL");
                    return false;
                }

                $entity = ProductEntity::findOne(C::_OR(C::eq('bling_id', $bling_produto_id), C::eq('codigo', $bling_produto_codigo))); // phpcs:ignore
            }

            if (!$entity || !$entity->product_id) {
                $this->dev5->debug("$prefix | |→ FAIL", "NO ID");
                $this->dev5->debug("$prefix |→ FAIL");
                return false;
            }

            $product_data = $entity->getProduct($entity->product_id, true);
            if (!$product_data) {
                $this->dev5->debug("$prefix | |→ FAIL", "NO DATA");
                $this->dev5->debug("$prefix |→ FAIL");
                return false;
            }

            $options = [];
            if ($entity->options) {
                foreach (explode(',', $entity->options) as $product_option_value_id) {
                    foreach ($product_data['product_option'] as $product_option) {
                        foreach ($product_option['values'] as $product_option_value) {
                            if ($product_option_value['product_option_value_id'] == $product_option_value_id) {
                                $options[] = [
                                    'product_option_id' => $product_option['product_option_id'],
                                    'product_option_value_id' => $product_option_value_id,
                                    'name' => $product_option['name'],
                                    'value' => $product_option_value['name'],
                                    'type' => $product_option['type']
                                ];
                                continue 3;
                            }
                        }
                    }

                    $this->dev5->debug("$prefix | |→ FAIL", "NO OPTION", $product_option_value_id);
                    $this->dev5->debug("$prefix |→ FAIL");
                    return false;
                }
            }

            $data['products'][] = [
                'product_id' => $entity->product_id,
                'master_id' => $this->getValue($product_data, 'master_id'), // 4x
                'name' => $this->getValue($product_data, 'name'),
                'model' => $this->getValue($product_data, 'model'),
                'quantity' => $quantity = $this->getValue($item, 'quantidade'),
                'price' => $price = $this->getValue($item, 'valor'),
                'total' => $quantity * $price,
                'tax' => 0,
                'reward' => $quantity * $this->getValue($product_data, 'reward'),
                'option' => $options
            ];

            $this->dev5->debug("$prefix | |→ SUCCESS", $entity->codigo);
        }

        $data['vouchers'] = $this->getValue($data, 'vouchers', []); // 2x, 3x

        $this->dev5->debug("$prefix | |→ SUCCESS", count($data['products']));

        return true;
    }

    /**
     * @param array $data
     * @param array $bling_order
     */
    private function importOrderNormalizeTotals(&$data, $bling_order)
    {
        $prefix_sort_order = ocver('3', '>=') ? 'total_' : '';
        $prefix_language = (ocver("3", ">=") ? 'extension/' : '') . (ocver("4", ">=") ? 'opencart/' : '');

        $data['totals'] = [];

        $this->load->language("{$prefix_language}total/sub_total");
        $data['totals'][] = [
            'extension' => 'opencart',
            'code' => 'sub_total',
            'title' => $this->language->get('text_sub_total'),
            'value' => $this->getValue($bling_order, 'totalProdutos'),
            'sort_order' => $this->config->get("{$prefix_sort_order}sub_total_sort_order")
        ];

        $this->load->language("{$prefix_language}total/shipping");
        $data['totals'][] = [
            'extension' => 'opencart',
            'code' => 'shipping',
            'title' => $this->getValue($data, ['shipping_method', 'name'], $this->getValue($data, 'shipping_method', 'Frete')), // phpcs:ignore
            'value' => $this->getValue($bling_order, ['transporte', 'frete']),
            'sort_order' => $this->config->get("{$prefix_sort_order}shipping_sort_order")
        ];

        if ($discount = $this->getValue($bling_order, ['desconto', 'valor'])) {
            $data['totals'][] = [
                'extension' => 'opencart',
                'code' => 'discount',
                'title' => 'Desconto',
                'value' => $discount,
                'sort_order' => max(
                    $this->config->get("{$prefix_sort_order}shipping_sort_order") + 1,
                    $this->config->get("{$prefix_sort_order}total_sort_order") - 1,
                    0
                )
            ];
        }

        $this->load->language("{$prefix_language}total/total");
        $data['totals'][] = [
            'extension' => 'opencart',
            'code' => 'total',
            'title' => $this->language->get('text_total'),
            'value' => $data['total'] = $this->getValue($bling_order, 'total'),
            'sort_order' => $this->config->get("{$prefix_sort_order}total_sort_order")
        ];

        array_multisort(
            array_column($data['totals'], 'sort_order'),
            SORT_ASC,
            array_keys($data['totals']),
            SORT_ASC,
            $data['totals']
        );
    }

    /**
     * @param int|null $order_id
     * @param array $data
     * @param string $prefix
     * @return int|false
     */
    private function importOrderSave($order_id, $data, $prefix = '')
    {
        $this->dev5->debug("$prefix • IMPORT-ORDER-SAVE", $order_id ? "UPDATE $order_id" : 'CREATE');

        $response = \Dev5\Client\Http\Json::post(
            $this->dev5->link->catalog("orderSave", "security={$this->dev5->extension->security}"),
            $post = [
                'order_id' => $order_id,
                'data' => $data
            ]
        );

        if ($response->code !== 201) {
            $this->dev5->debug("$prefix |→ FAIL", ['i' => $post, 'o' => [$response->code, $response->content]]);
            return false;
        }

        if (empty($response->content['order_id'])) {
            $this->dev5->debug("$prefix |→ FAIL", "NO-ORDER-ID");
            return false;
        }

        $this->dev5->debug("$prefix |→ SUCCESS", $response->content['order_id']);
        return (int)$response->content['order_id'];
    }

    /**
     * @param int $order_id
     * @param int $bling_status_id
     * @param string $prefix
     * @return bool
     */
    private function importOrderStatus($order_id, $bling_status_id, $prefix = '', &$error = null)
    {
        $this->dev5->debug("$prefix • IMPORT-ORDER-STATUS", $order_id, "status=$bling_status_id");

        $ignore_id = $this->dev5->field('ei')['export_order_ignore_id'];
        if ($ignore_id && $ignore_id > $order_id) {
            $this->dev5->debug("$prefix |→ SKIP", "IGNORED-ID");
            return true;
        }

        $order = @$this->getOrder($order_id);
        if (!$order) {
            $error = ['message' => 'FAIL, NO-DATA', 'file' => __FILE__ . ':' . __LINE__];
            $this->dev5->debug("$prefix |→ FAIL", "NO-DATA");
            return false;
        }

        $ignore_date = $this->dev5->field('ei')['export_order_ignore_date'];
        if ($ignore_date && strtotime($ignore_date) > strtotime($order['date_added'])) {
            $this->dev5->debug("$prefix |→ SKIP", "IGNORED-DATE");
            return true;
        }

        $b2o = $this->dev5->field('order_status');
        if (empty($b2o[$bling_status_id]['status_id'])) {
            $this->dev5->debug("$prefix |→ SKIP", "IGNORED-STATUS[$bling_status_id]");
            return true;
        }

        if ($order['order_status_id'] == $b2o[$bling_status_id]['status_id']) {
            $this->dev5->debug("$prefix |→ SKIP", "SAME-STATUS");
            return true;
        }

        $response = \Dev5\Client\Http\Json::post(
            $this->dev5->link->catalog("orderSave", "security={$this->dev5->extension->security}"),
            $post = [
                'order_id' => $order_id,
                'order_status_id' => $b2o[$bling_status_id]['status_id'],
                'comment' => "Integração ERP Bling API V3",
                'notify' => !empty($b2o[$bling_status_id]['notify']) ? 1 : 0,
                'override' => 0
            ]
        );

        if ($response->code !== 204) {
            $error = ['message' => 'FAIL, SAVE-STATUS', 'file' => __FILE__ . ':' . __LINE__, 'debug' => Api::getInstance()->last()]; // phpcs:ignore
            $this->dev5->debug("$prefix |→ FAIL", "SAVE-STATUS", Api::getInstance()->last());
            return false;
        }

        $this->dev5->debug("$prefix |→ SUCCESS");
        return true;
    }

    /**
     * @param EIEntity\Payload $payload
     * @param string $prefix
     * @return bool
     */
    public function importOrders($payload, $prefix = '')
    {
        $this->dev5->debug("$prefix • IMPORT-ORDERS", $payload->cursor);

        $filter = [];

        !empty($payload->input['status'])
        && $filter['idsSituacoes'] = $payload->input['status'];

        !empty($payload->input['numero_loja'])
        && $filter['numerosLojas'] = $payload->input['numero_loja'];

        !empty($payload->input['date_added']['start'])
        && $filter['dataInicial'] = $payload->input['date_added']['start'];

        !empty($payload->input['date_added']['end'])
        && $filter['dataFinal'] = $payload->input['date_added']['end'];

        !empty($payload->input['date_modified']['start'])
        && $filter['dataAlteracaoInicial'] = "{$payload->input['date_modified']['start']} 00:00:00";

        !empty($payload->input['date_modified']['end'])
        && $filter['dataAlteracaoFinal'] = "{$payload->input['date_modified']['end']} 23:59:59";

        !empty($payload->input['channel_id'])
        && $filter['idLoja'] = $payload->input['channel_id'];

        if (empty($payload->input['bling_id'])) {
            $limit = 100;

            $page = ['start' => $payload->cursor + 1, 'limit' => $limit];

            $rows = Api::getInstance()->getPedidosVenda($filter, ['page' => $page]);
        } else {
            $limit = 1;
            $rows = [];

            if ($payload->cursor < count($payload->input['bling_id'])) {
                $row = Api::getInstance()->getPedidoVenda($payload->input['bling_id'][$payload->cursor]);
                $row['full'] = true;

                // phpcs:disable
                $row && !empty($filter['idsSituacoes']) && !in_array($row['situacao']['id'], $filter['idsSituacoes']) && $row = null;
                $row && !empty($filter['numerosLojas']) && !in_array($row['numeroLoja'], $filter['numerosLojas']) && $row = null;
                $row && !empty($filter['dataInicial']) && strtotime($row['data']) < strtotime("{$filter['dataInicial']} 00:00:00") && $row = null;
                $row && !empty($filter['dataFinal']) && strtotime($row['data']) > strtotime("{$filter['dataFinal']} 23:59:59") && $row = null;
                $row && !empty($filter['dataAlteracaoInicial']) && strtotime($row['dataAlteracao']) < strtotime($filter['dataAlteracaoInicial']) && $row = null;
                $row && !empty($filter['dataAlteracaoFinal']) && strtotime($row['dataAlteracao']) > strtotime($filter['dataAlteracaoFinal']) && $row = null;
                $row && !empty($filter['idLoja']) && $row['loja']['id'] != $filter['idLoja'] && $row = null;
                // phpcs:enable
                $rows[] = $row;
            }
        }

        foreach ($rows as $row) {
            if (!$row) {
                continue;
            }

            $total = $this->currency->convert($row['total'], $this->dev5->field('currency_code'), $this->config->get('config_currency')); // phpcs:ignore
            $total = $this->currency->format($total, $this->config->get('config_currency'));

            EIEntity::fromArray([
                'id' => $row['id'],
                'label' => "#$row[id] {$row['contato']['nome']} $total"
            ])->save(['returning' => false]);
            $payload->total++;

            $this->dev5->debug("$prefix | • $row[id] → PENDING");
        }

        $count = count($rows);

        $this->dev5->debug("$prefix |→ SUCCESS", "$count/$limit");

        return $count < $limit;
    }

    /**
     * @param EIEntity\Payload $payload
     * @param string $prefix
     * @return bool
     */
    public function importOrdersApprove($payload, $prefix = '')
    {
        $this->dev5->debug("$prefix • IMPORT-ORDERS-APPROVE", $payload->cursor);

        if ($entity = EIEntity::findOne(['status' => EIEntity::PENDING])) {
            $this->dev5->debug("$prefix | • IMPORT-ORDER-APPROVE", $entity->id);
            QueueEntity::publish("import_order:$entity->id", ['event' => 'ei'], QueueEntity::PROCESSING, $queue);

            $entity->status = $this->importOrder($entity->id, "$prefix | |", $error) ? EIEntity::SUCCESS : EIEntity::FAIL; // phpcs:ignore
            $entity->save(['returning' => false]);

            ($queue->status = ($entity->status === EIEntity::SUCCESS ? QueueEntity::SUCCESS : QueueEntity::FAIL))
            === QueueEntity::FAIL && $queue->payload = array_merge($queue->payload, [
                'error' => !empty($error) ? $error : Api::getInstance()->error(),
                'debug' => Api::getInstance()->last()
            ]);
            $queue->save(['returning' => false]);

            $payload->total++;
            $this->dev5->debug("$prefix | |→ " . ($entity->status === EIEntity::SUCCESS ? 'SUCCESS' : 'FAIL'));
        }

        $finished = !$entity || EIEntity::findTotal(['status' => EIEntity::PENDING]) < 1;

        $this->dev5->debug("$prefix |→ " . ($finished ? 'FINISHED' : 'NEXT'));

        return $finished;
    }

    /**
     * @param array $data
     * @param string $prefix
     * @return bool
     */
    public function importCallback($data, $prefix = '')
    {
        $this->dev5->debug("$prefix • IMPORT-ORDER-CALLBACK", $data['id']);
        QueueEntity::publish("callback_order_status:$data[id]", ['event' => 'callback'], QueueEntity::PROCESSING, $queue); // phpcs:ignore
        $queue->status = QueueEntity::FAIL;

        try {
            if (!($bling_order = current(Api::getInstance()->getPedidosVenda(['numero' => $data['numero']])))) {
                $this->dev5->debug("$prefix |→ FAIL", "NO-DATA");
                return false;
            }

            $entity = self::findOne(['bling_id' => $bling_order['id']]) ?: new self();
            if (!$entity->order_id) {
                $this->dev5->debug("$prefix | • FULL-IMPORT");
                $result = $this->importOrder($bling_order['id'], "$prefix |");
            } else {
                $this->dev5->debug("$prefix | • IMPORT-ONLY-STATUS");
                $result = $this->importOrderStatus($entity->order_id, $this->getValue($bling_order, ['situacao', 'id']), "$prefix |"); // phpcs:ignore
            }

            if ($result) {
                $this->dev5->debug("$prefix |→ SUCCESS");
                $queue->status = QueueEntity::SUCCESS;
                return true;
            }

            $this->dev5->debug("$prefix |→ FAIL");
        } catch (\Exception $e) {
            $error = "$e";
            $this->dev5->debug("$prefix |→ FAIL", "$e");
        } catch (\Throwable $e) {
            $error = "$e";
            $this->dev5->debug("$prefix |→ FAIL", "$e");
        } finally {
            $queue->status === QueueEntity::FAIL && $queue->payload = array_merge($queue->payload, [
                'error' => !empty($error) ? $error : Api::getInstance()->error(),
                'debug' => Api::getInstance()->last()
            ]);
            $queue->save(['returning' => false]);
        }
        return false;
    }
}
