<?php

/**
 * This file is part of Dev5 Integração ERP Bling API V3 for OpenCart
 *
 * @author    Dev5™ <developer@dev5.com.br>
 * @copyright Dev5™ 2025. Todos os direitos reservados.
 * @license   Distribuição/reprodução não autorizada estará sujeita às penas das leis 9609/98 e 9610/98
 * @link      https://dev5.com.br/integracao-erp-bling-api-v3
 * @version   4.0.6
 */

namespace Dev5\Library\Bling;

/** @Dev5\Orm\Entity(['name' => DB_PREFIX . 'd5bling3_category', 'auto_migrate' => false]) */
final class CategoryEntity extends BaseEntity
{
    /**
     * @var int
     * @Dev5\Orm\Field(['type' => 'int', 'primary' => true])
     */
    public $category_id;

    /**
     * @var string|null
     * @Dev5\Orm\Field(['type' => 'varchar', 'length' => 64, 'index' => true, 'nullable' => true])
     */
    public $bling_id;

    /**
     * @param int $category_id
     * @return array|null
     */
    public function getCategory($category_id)
    {
        $data = $this->dev5->db->query("
            SELECT
                cd.name,
                c.parent_id,
                c.sort_order,
                c.column,
                c.image,
                c.status,
                {$this->getColumnCategoryTop()} AS `top`,
                (
                    SELECT
                        GROUP_CONCAT(cd.name ORDER BY cp.level SEPARATOR ' > ')
                    FROM
                        " . DB_PREFIX . "category_path cp
                    INNER JOIN
                        " . DB_PREFIX . "category_description cd
                        ON cd.category_id = cp.path_id
                        AND cd.language_id = :language_id
                    WHERE
                        cp.category_id = c.category_id
                ) AS `full_name`
            FROM
                " . DB_PREFIX . "category c
            INNER JOIN
                " . DB_PREFIX . "category_description cd
                ON c.category_id = cd.category_id
                AND cd.language_id = :language_id
            WHERE
                c.category_id = :category_id
        ", [
            'category_id' => $category_id,
            'language_id' => (int)$this->dev5->field('language_id')
        ])->row();

        if (!$data) {
            return null;
        }

        $data['name'] = html_entity_decode($data['name']);

        $data['full_name'] = html_entity_decode($data['full_name']);

        $data['category_store'] = array_map('intval', array_column($this->dev5->db->query("
            SELECT
                c2s.store_id
            FROM
                " . DB_PREFIX . "category_to_store c2s
            WHERE
                c2s.category_id = :category_id
        ", [
            'category_id' => $category_id
        ])->all(), 'store_id'));

        $data['category_description'] = array_column($this->dev5->db->query("
            SELECT
                cd.language_id,
                cd.name,
                cd.description,
                cd.meta_title,
                cd.meta_description,
                cd.meta_keyword
            FROM
                " . DB_PREFIX . "category_description cd
            WHERE
                cd.category_id = :category_id
        ", [
            'category_id' => $category_id
        ])->all(), null, 'language_id');
        foreach ($data['category_description'] as &$description) {
            $description['name'] = html_entity_decode($description['name']);
            $description['description'] = html_entity_decode($description['description']);
            $description['meta_title'] = html_entity_decode($description['meta_title']);
            $description['meta_description'] = html_entity_decode($description['meta_description']);
            $description['meta_keyword'] = html_entity_decode($description['meta_keyword']);
            unset($description['language_id']);
        }

        return $data;
    }

    /**
     * @var array<string, array>
     */
    private static $cache_bling_categories = null;

    /**
     * @return array<string, array>
     */
    public function getBlingCategories()
    {
        if (self::$cache_bling_categories === null) {
            self::$cache_bling_categories = array_column(Api::getInstance()->getCategoriasProdutos(), null, 'id');
        }
        return self::$cache_bling_categories;
    }

    /**
     * @param string $bling_id
     * @return array|null
     */
    public function getCacheBlingCategoria($bling_id)
    {
        $this->getBlingCategories();
        return isset(self::$cache_bling_categories[$bling_id]) ? self::$cache_bling_categories[$bling_id] : null;
    }

    /**
     * @param string $bling_id
     * @param array|null $data
     * @return void
     */
    public function setCacheBlingCategoria($bling_id, $data)
    {
        $this->getBlingCategories();
        if (null === $data) {
            unset(self::$cache_bling_categories[$bling_id]);
        } else {
            $data['id'] = $bling_id;
            self::$cache_bling_categories[$bling_id] = $data;
        }
    }

    /**
     * @param string $descricao
     * @param string $prefix
     * @return bool
     */
    private function exportCategoryChannel($descricao, $prefix = '')
    {
        $this->dev5->debug("$prefix • EXPORT-CATEGORY-CHANNEL", $this->category_id);

        $descricao = html_entity_decode($descricao);

        $bling_categoria = $this->getCacheBlingCategoria($this->bling_id);

        if (!empty($bling_categoria['loja_vinculo_id'])) {
            $this->dev5->debug("$prefix | • MEMORY");
            $this->dev5->debug("$prefix |→ SUCCESS", $bling_categoria['loja_vinculo_id']);
            return true;
        }

        $main_channel_id = $this->dev5->field('channel_id');

        $bling_loja = current(Api::getInstance()->getCategoriasProdutosLojas([
            'idLoja' => $main_channel_id,
            'idCategoriaProduto' => $this->bling_id
        ]));

        if (!$bling_loja) {
            $this->dev5->debug("$prefix | • CREATE");
            $bling_loja_vinculo_id = Api::getInstance()->postCategoriaProdutoLoja([
                'loja' => ['id' => $main_channel_id],
                'descricao' => $descricao,
                'codigo' => $this->category_id,
                'categoriaProduto' => ['id' => $this->bling_id]
            ]);
        } elseif ($bling_loja['codigo'] == $this->category_id && $bling_loja['descricao'] == $descricao) {
            $this->dev5->debug("$prefix | • SKIP", "SAME DATA");
            $bling_loja_vinculo_id = $bling_loja['id'];
        } else {
            $this->dev5->debug("$prefix | • UPDATE");
            $bling_loja_vinculo_id = Api::getInstance()->putCategoriaProdutoLoja($bling_loja['id'], [
                'loja' => ['id' => $main_channel_id],
                'descricao' => $descricao,
                'codigo' => $this->category_id,
                'categoriaProduto' => ['id' => $this->bling_id]
            ]);
        }

        if (!$bling_loja_vinculo_id) {
            $this->dev5->debug("$prefix | |→ FAIL", Api::getInstance()->error());
            $this->dev5->debug("$prefix |→ FAIL");
            return false;
        }

        $bling_categoria['loja_vinculo_id'] = $bling_loja_vinculo_id;
        $this->setCacheBlingCategoria($this->bling_id, $bling_categoria);

        $this->dev5->debug("$prefix | |→ SUCCESS");
        $this->dev5->debug("$prefix |→ SUCCESS", $bling_loja_vinculo_id);
        return true;
    }

    /**
     * @param array $payload
     * @param string $prefix
     * @return string|false
     */
    public function exportCategory($payload, $prefix = '')
    {
        $this->dev5->debug("$prefix • EXPORT-CATEGORY", $this->category_id);

        $category = $this->getCategory($this->category_id);
        if (!$category) {
            $this->dev5->debug("$prefix |→ FAIL", "NO-DATA");
            return false;
        }

        // phpcs:disable
        $bling_categoria_pai_id = empty($category['parent_id'])
            ? false
            : self::getInstance($category['parent_id'])->exportCategory($payload, "$prefix |");

        $data = $this->filterNullable([
            'descricao' => html_entity_decode($category['name']),
            'categoriaPai' => ['id' => $bling_categoria_pai_id ?: 0]
        ]);

        $bling_categoria = array_find($this->getBlingCategories(), static function ($category) use ($data) {
            return $data['descricao'] === $category['descricao']
                && $data['categoriaPai']['id'] == $category['categoriaPai']['id'];
        });
        // phpcs:enable

        if ($bling_categoria) {
            $this->dev5->debug("$prefix | • SKIP", "SAME DATA");
            $this->dev5->debug("$prefix | |→ SUCCESS", $bling_categoria['id']);
            $this->bling_id = $bling_categoria['id'];
        } elseif ($this->bling_id && ($bling_categoria = $this->getCacheBlingCategoria($this->bling_id))) {
            if ($data['categoriaPai']['id'] != $bling_categoria['categoriaPai']['id']) {
                $this->dev5->debug("$prefix | • RECREATE");
                $this->exportDeleteCategory($payload, "$prefix |");
                $this->bling_id = Api::getInstance()->postCategoriaProduto($data);
            } else {
                $this->dev5->debug("$prefix | • UPDATE");
                $this->bling_id = Api::getInstance()->putCategoriaProduto($this->bling_id, $data);
            }
        } else {
            $this->dev5->debug("$prefix | • CREATE");
            $this->bling_id = Api::getInstance()->postCategoriaProduto($data);
        }

        if (!$this->bling_id) {
            $this->dev5->debug("$prefix | |→ FAIL", Api::getInstance()->error());
            $this->dev5->debug("$prefix |→ FAIL");
            return false;
        }

        $this->setCacheBlingCategoria($this->bling_id, $data);

        if (!$this->exportCategoryChannel($category['full_name'], "$prefix |")) {
            $this->setCacheBlingCategoria($this->bling_id, null);
            $this->dev5->debug("$prefix |→ FAIL");
            return false;
        }

        $this->save(['returning' => false]);
        $this->dev5->debug("$prefix |→ SUCCESS", $this->bling_id);

        return $this->bling_id;
    }

    /**
     * @param array $payload
     * @param string $prefix
     * @return bool
     */
    public function exportDeleteCategory($payload, $prefix = '')
    {
        $this->dev5->debug("$prefix • DELETE-CATEGORY", $this->category_id);

        if (!$this->bling_id) {
            $this->dev5->debug("$prefix |→ SUCCESS", "NO-DATA");
            return true;
        }

        $children = array_filter($this->getBlingCategories(), function ($category) {
            return $category['categoriaPai']['id'] == $this->bling_id;
        }) ?: [];

        foreach ($children as $child) {
            $child_entity = self::findOne(['bling_id' => $child['id']]);
            $child_entity && !$child_entity->exportDeleteCategory($payload, "$prefix |");
        }

        if (!Api::getInstance()->deleteCategoriaProduto($this->bling_id)) {
            $this->dev5->debug("$prefix | |→ FAIL", Api::getInstance()->error());
            return false;
        }

        $this->setCacheBlingCategoria($this->bling_id, null);
        self::delete($this->category_id);

        $this->dev5->debug("$prefix |→ SUCCESS");
        return true;
    }

    /**
     * @param EIEntity\Payload $payload
     * @param string $prefix
     * @return bool
     */
    public function exportCategories($payload, $prefix = '')
    {
        $this->dev5->debug("$prefix • EXPORT-CATEGORIES", $payload->cursor);

        $where = [];
        $values = ['language_id' => (int)$this->dev5->field('language_id')];

        if (!empty($payload->input['id']) || !empty($payload->input['name'])) {
            if (!empty($payload->input['id'])) {
                $category_ids = $payload->input['id'];
            }

            if (!empty($payload->input['name'])) {
                $temp_ids = array_column($payload->input['name'], 'value');
                $category_ids = isset($category_ids)
                    ? array_intersect($category_ids, $temp_ids)
                    : $temp_ids;
            }

            $where[] = "c.category_id IN :category_id";
            $values['category_id'] = empty($category_ids) ? [-1] : array_unique($category_ids);
        }

        if (array_key_exists('status', $payload->input) && strlen($payload->input['status']) > 0) {
            $where[] = "c.status = :status";
            $values['status'] = $payload->input['status'];
        }

        if (!empty($payload->input['date_added']['start']) && !empty($payload->input['date_added']['end'])) {
            $where[] = "c.date_added BETWEEN :date_added_start AND :date_added_end";
            $values['date_added_start'] = "{$payload->input['date_added']['start']} 00:00:00";
            $values['date_added_end'] = "{$payload->input['date_added']['end']} 23:59:59";
        } elseif (!empty($payload->input['date_added']['start'])) {
            $where[] = "c.date_added >= :date_added_start";
            $values['date_added_start'] = "{$payload->input['date_added']['start']} 00:00:00";
        } elseif (!empty($payload->input['date_added']['end'])) {
            $where[] = "c.date_added <= :date_added_end";
            $values['date_added_end'] = "{$payload->input['date_added']['end']} 23:59:59";
        }

        if (!empty($payload->input['date_modified']['start']) && !empty($payload->input['date_modified']['end'])) {
            $where[] = "c.date_modified BETWEEN :date_modified_start AND :date_modified_end";
            $values['date_modified_start'] = "{$payload->input['date_modified']['start']} 00:00:00";
            $values['date_modified_end'] = "{$payload->input['date_modified']['end']} 23:59:59";
        } elseif (!empty($payload->input['date_modified']['start'])) {
            $where[] = "c.date_modified >= :date_modified_start";
            $values['date_modified_start'] = "{$payload->input['date_modified']['start']} 00:00:00";
        } elseif (!empty($payload->input['date_modified']['end'])) {
            $where[] = "c.date_modified <= :date_modified_end";
            $values['date_modified_end'] = "{$payload->input['date_modified']['end']} 23:59:59";
        }

        $where = empty($where) ? '' : 'WHERE ' . implode(' AND ', $where);
        $limit = 100;
        $cursor = $payload->cursor * $limit;

        $rows = $this->dev5->db->query("
            SELECT
                c.category_id,
                (
                    SELECT
                        GROUP_CONCAT(cd.name ORDER BY cp.level SEPARATOR ' > ')
                    FROM
                        " . DB_PREFIX . "category_path cp
                    INNER JOIN
                        " . DB_PREFIX . "category_description cd
                        ON cd.category_id = cp.path_id
                        AND cd.language_id = :language_id
                    WHERE
                        cp.category_id = c.category_id
                ) AS `full_name`
            FROM
                " . DB_PREFIX . "category c
            $where
            ORDER BY c.category_id
            LIMIT $cursor, $limit;
        ", $values)->all();

        foreach ($rows as $row) {
            EIEntity::fromArray([
                'id' => $row['category_id'],
                'label' => "#$row[category_id] $row[full_name]"
            ])->save(['returning' => false]);
            $payload->total++;

            $this->dev5->debug("$prefix | • $row[category_id] → PENDING");
        }

        $count = count($rows);

        $this->dev5->debug("$prefix |→ SUCCESS", "$count/$limit");

        return $count < $limit;
    }

    /**
     * @param EIEntity\Payload $payload
     * @param string $prefix
     * @return bool
     */
    public function exportCategoriesApprove($payload, $prefix = '')
    {
        $this->dev5->debug("$prefix • EXPORT-CATEGORIES-APPROVE", $payload->cursor);

        $limit = 100;
        $entities = EIEntity::findMany(['status' => EIEntity::PENDING], ['limit' => $limit]);

        foreach ($entities as $entity) {
            $this->dev5->debug("$prefix | • EXPORT-CATEGORY-APPROVE", $entity->id);

            QueueEntity::publish("export_category:$entity->id", ['event' => 'ei']);
            $payload->total++;

            $entity->status = EIEntity::QUEUE;
            $entity->save(['returning' => false]);

            $this->dev5->debug("$prefix | |→ QUEUE");
        }

        $count = count($entities);

        $this->dev5->debug("$prefix |→ SUCCESS", "$count/$limit");

        return $count < $limit;
    }

    /**
     * @param string $bling_id
     * @param EIEntity\Payload|null $payload
     * @param string $prefix
     * @return int|false
     */
    public function importCategory($bling_id, $payload = null, $prefix = '', &$error = null)
    {
        $this->dev5->debug("$prefix • IMPORT-CATEGORY", $bling_id);

        $bling_categoria = $this->getCacheBlingCategoria($bling_id);

        if (!$bling_categoria) {
            $error = ['message' => 'FAIL, NO-DATA', 'file' => __FILE__ . ':' . __LINE__, 'debug' => Api::getInstance()->last()]; // phpcs:ignore
            $this->dev5->debug("$prefix |→ FAIL", "NO-DATA", Api::getInstance()->error());
            return false;
        }

        if (isset($bling_categoria['category_id'])) {
            $this->dev5->debug("$prefix | • CACHED");
            $this->dev5->debug("$prefix |→ SUCCESS", $bling_categoria['category_id']);
            return $bling_categoria['category_id'];
        }

        $entity = self::findOne(['bling_id' => $bling_id]) ?: new self();
        $entity->bling_id = $bling_id;

        $parent_id = 0;
        if ($bling_categoria_pai_id = $this->getValue($bling_categoria, ['categoriaPai', 'id'])) {
            $this->dev5->debug("$prefix | • IMPORT-CATEGORY-PARENT", $bling_categoria_pai_id);
            if (!($parent_id = $this->importCategory($bling_categoria_pai_id, $payload, "$prefix | |", $error))) {
                $this->dev5->debug("$prefix | |→ FAIL");
                $this->dev5->debug("$prefix |→ FAIL");
                return false;
            }
            $this->dev5->debug("$prefix | |→ SUCCESS", $parent_id);
        }

        if ($entity->category_id && (!($current = $this->getCategory($entity->category_id)))) {
            $entity->category_id = 0;
        }

        // phpcs:disable
        $current = empty($current) ? [] : $current;

        $data = $current;
        $data['parent_id'] = (int)$parent_id;
        $data['image'] = $this->getValue($data, 'image', '');
        $data['top'] = $this->getValue($data, 'top', '');
        $data['column'] = $this->getValue($data, 'column', 1);
        $data['sort_order'] = $this->getValue($data, 'sort_order', 0);
        $data['status'] = $this->getValue($data, 'status', 1);
        $data['category_store'] = $this->getValue($data, 'category_store', array_merge([0], array_column($this->model_setting_store->getStores(), 'store_id')));
        foreach ($this->getLanguages() as $language_id) {
            $data['category_description'][$language_id]['name'] = $this->getValue($bling_categoria, 'descricao');
            $data['category_description'][$language_id]['description'] = $this->getValue($data, ['category_description', $language_id, 'description'], '');
            $data['category_description'][$language_id]['meta_title'] = $this->getValue($bling_categoria, 'descricao');
            $data['category_description'][$language_id]['meta_description'] = $this->getValue($data, ['category_description', $language_id, 'meta_description'], '');
            $data['category_description'][$language_id]['meta_keyword'] = $this->getValue($data, ['category_description', $language_id, 'meta_keyword'], '');
        }
        // phpcs:enable

        try {
            self::ignore(true);
            if (!array_multidiff($data, $current)) {
                $this->dev5->debug("$prefix | • SKIP", "SAME DATA");
            } elseif (!$entity->category_id) {
                $this->dev5->debug("$prefix | • CREATE");
                $entity->category_id = $this->model_catalog_category->addCategory($data);
            } else {
                $this->dev5->debug("$prefix | • UPDATE");
                $this->model_catalog_category->editCategory($entity->category_id, $data);
            }
        } catch (\Exception $e) {
            $error = "$e";
            $this->dev5->debug("$prefix | |→ FAIL", "$e");
            $this->dev5->debug("$prefix |→ FAIL");
            return false;
        } catch (\Throwable $e) {
            $error = "$e";
            $this->dev5->debug("$prefix | |→ FAIL", "$e");
            $this->dev5->debug("$prefix |→ FAIL");
            return false;
        } finally {
            self::ignore(false);
        }

        $bling_categoria['category_id'] = $entity->category_id;
        $this->setCacheBlingCategoria($bling_id, $bling_categoria);
        $entity->save(['returning' => false]);

        $this->dev5->debug("$prefix | |→ SUCCESS");
        $this->dev5->debug("$prefix |→ SUCCESS", $entity->category_id);
        return $entity->category_id;
    }

    /**
     * @param EIEntity\Payload $payload
     * @param string $prefix
     * @return bool
     */
    public function importCategories($payload, $prefix = '')
    {
        $this->dev5->debug("$prefix • IMPORT-CATEGORIES", $payload->cursor);

        $rows = Api::getInstance()->getCategoriasProdutosNormalized();

        !empty($payload->input['bling_id'])
        && $rows = array_intersect_key($rows, array_flip($payload->input['bling_id']));

        !empty($payload->input['name'])
        && $rows = array_intersect_key($rows, array_column($payload->input['name'], null, 'value'));

        foreach ($rows as $bling_categoria_id => $nome) {
            EIEntity::fromArray([
                'id' => $bling_categoria_id,
                'label' => "#$bling_categoria_id $nome"
            ])->save(['returning' => false]);
            $payload->total++;

            $this->dev5->debug("$prefix | • $bling_categoria_id → PENDING");
        }

        $this->dev5->debug("$prefix |→ SUCCESS");
        return true;
    }

    /**
     * @param EIEntity\Payload $payload
     * @param string $prefix
     * @return bool
     */
    public function importCategoriesApprove($payload, $prefix = '')
    {
        $this->dev5->debug("$prefix • IMPORT-CATEGORIES-APPROVE", $payload->cursor);

        if ($entity = EIEntity::findOne(['status' => EIEntity::PENDING])) {
            $this->dev5->debug("$prefix | • IMPORT-CATEGORY-APPROVE", $entity->id);
            QueueEntity::publish("import_category:$entity->id", ['event' => 'ei'], QueueEntity::PROCESSING, $queue);

            $entity->status = $this->importCategory($entity->id, $payload, "$prefix | |", $error) ? EIEntity::SUCCESS : EIEntity::FAIL; // phpcs:ignore
            $entity->save(['returning' => false]);

            ($queue->status = ($entity->status === EIEntity::SUCCESS ? QueueEntity::SUCCESS : QueueEntity::FAIL))
            === QueueEntity::FAIL && $queue->payload = array_merge($queue->payload, [
                'error' => !empty($error) ? $error : Api::getInstance()->error(),
                'debug' => Api::getInstance()->last()
            ]);
            $queue->save(['returning' => false]);

            $payload->total++;
            $this->dev5->debug("$prefix | |→ " . ($entity->status === EIEntity::SUCCESS ? 'SUCCESS' : 'FAIL'));
        }

        $this->dev5->debug("$prefix |→ SUCCESS");

        return !$entity || EIEntity::findTotal(['status' => EIEntity::PENDING]) < 1;
    }
}
