<?php
/**
 * Box packing (3D bin packing, knapsack problem).
 *
 * @author Doug Wright
 */
namespace D5WEXT\Packing\DVDoug\BoxPacker;

use function array_map;
use ArrayIterator;
use function count;
use Countable;
use IteratorAggregate;
use Traversable;
use function usort;

/**
 * List of packed items, ordered by volume.
 *
 * @author Doug Wright
 */
class PackedItemList implements Countable, IteratorAggregate
{
    /**
     * List containing items.
     *
     * @var PackedItem[]
     */
    private $list = [];

    private $weight = 0;

    /**
     * Has this list already been sorted?
     *
     * @var bool
     */
    private $isSorted = false;

    /**
     * @param PackedItem $item
     * @return void
     */
    public function insert(PackedItem $item)
    {
        $this->list[] = $item;
        $this->weight += $item->getItem()->getWeight();
    }

    /**
     * @return Traversable|PackedItem[]
     */
    #[\ReturnTypeWillChange]
    public function getIterator()
    {
        if (!$this->isSorted) {
            usort($this->list, [$this, 'compare']);
            $this->isSorted = true;
        }

        return new ArrayIterator($this->list);
    }

    /**
     * Number of items in list.
     * @return int
     */
    #[\ReturnTypeWillChange]
    public function count()
    {
        return count($this->list);
    }

    /**
     * Get copy of this list as a standard PHP array.
     *
     * @internal
     *
     * @return Item[]
     */
    public function asItemArray()
    {
        return array_map(function (PackedItem $packedItem) {
            return $packedItem->getItem();
        }, $this->list);
    }

    /**
     * Get total volume of these items.
     * @return int
     */
    public function getVolume()
    {
        $volume = 0;

        foreach ($this->list as $item) {
            $volume += $item->getVolume();
        }

        return $volume;
    }

    /**
     * Get total weight of these items.
     * @return int
     */
    public function getWeight()
    {
        return $this->weight;
    }

    /**
     * @param PackedItem $itemA
     * @param PackedItem $itemB
     * @return int
     */
    private function compare(PackedItem $itemA, PackedItem $itemB)
    {
        $itemAVolume = $itemA->getItem()->getWidth() * $itemA->getItem()->getLength() * $itemA->getItem()->getDepth();
        $itemBVolume = $itemB->getItem()->getWidth() * $itemB->getItem()->getLength() * $itemB->getItem()->getDepth();

        return mixCmp($itemBVolume, $itemAVolume) ?: mixCmp($itemB->getItem()->getWeight(), $itemA->getItem()->getWeight());
    }
}
