<?php // phpcs:ignore

/**
 * This file is part of Dev5 Integração ERP Bling API V3 for OpenCart
 *
 * @author    Dev5™ <developer@dev5.com.br>
 * @copyright Dev5™ 2025. Todos os direitos reservados.
 * @license   Distribuição/reprodução não autorizada estará sujeita às penas das leis 9609/98 e 9610/98
 * @link      https://dev5.com.br/integracao-erp-bling-api-v3
 * @version   4.0.6
 */

namespace Dev5\OpenCart\Controller\Bling;

use Dev5\Library\Bling\Api;
use Dev5\Library\Bling\CategoryEntity;
use Dev5\Library\Bling\OrderEntity;
use Dev5\Library\Bling\ProductEntity;
use Dev5\Library\Bling\QueueEntity;
use Dev5\Library\RWin\RWin;
use Exception;
use Throwable;

use function Dev5\Opencart\ocver;

abstract class Catalog extends \Dev5\OpenCart
{
    /**
     * @param string $event
     * @param int $order_id
     * @return void
     */
    public function orderEvent($event, $order_id = null)
    {
        if (OrderEntity::ignore()) {
            return;
        }

        if (!$this->dev5->status()) {
            return;
        }

        if (empty($order_id) || !is_numeric($order_id)) {
            return;
        }

        if (!in_array($event, @$this->dev5->field('ei')['export_order'] ?: [])) {
            return;
        }

        QueueEntity::publish("export_order:$order_id", ['event' => $event]);
    }

    public function orderSave()
    {
        try {
            if (@$this->request->get['security'] !== $this->dev5->extension->security) {
                http_response_code(401);
                exit;
            }

            if (!($order_save = Api::getInstance()->getOrderSave())) {
                http_response_code(400);
                exit;
            }

            if (!empty($order_save['data'])) {
                $order_id = isset($order_save['order_id']) ? $order_save['order_id'] : null;
                $data = $order_save['data'];

                try {
                    self::ignore(true);
                    if (!$order_id) {
                        $order_id = $this->model_checkout_order->addOrder($data);
                    } else {
                        $this->model_checkout_order->editOrder($order_id, $data);
                    }
                } catch (Exception $e) {
                    $order_id = null;
                } catch (Throwable $e) {
                    $order_id = null;
                } finally {
                    self::ignore(false);
                }

                if (!$order_id) {
                    http_response_code(500);
                    header('Content-Type: application/json; charset=utf-8');
                    echo json_encode(['error' => isset($e) ? "$e" : 'Internal Server Error'], JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE); // phpcs:ignore
                    exit;
                }

                if (!empty($data['date_added'])) {
                    $this->dev5->db->queryUpdate(DB_PREFIX . 'order', [
                        'values' => ['date_added' => $data['date_added'],],
                        'where' => ['order_id' => $order_id]
                    ]);
                }

                http_response_code(201);
                header('Content-Type: application/json; charset=utf-8');
                echo json_encode(['order_id' => $order_id], JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
                exit;
            }

            if (!empty($order_save['order_id']) && !empty($order_save['order_status_id'])) {
                $order_id = intval($order_save['order_id']);
                $order_status_id = intval($order_save['order_status_id']);
                $comment = isset($order_save['comment']) ? strval($order_save['comment']) : '';
                $notify = !!intval(isset($order_save['notify']) ? $order_save['notify'] : '0');
                $overwrite = !!intval(isset($order_save['overwrite']) ? $order_save['overwrite'] : '0');

                try {
                    self::ignore(true);
                    $this->model_checkout_order->{ocver('4', '>=') ? 'addHistory' : 'addOrderHistory'}(
                        $order_id,
                        $order_status_id,
                        $comment,
                        $notify,
                        $overwrite
                    );
                } catch (Exception $e) {
                } catch (Throwable $e) {
                } finally {
                    self::ignore(false);
                }

                if (isset($e)) {
                    http_response_code(500);
                    header('Content-Type: application/json; charset=utf-8');
                    echo json_encode(['error' => "$e"], JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE); // phpcs:ignore
                    exit;
                }

                http_response_code(204);
                exit;
            }

            http_response_code(400);
            exit;
        } finally {
            @$this->session->destroy();
        }
    }

    /**
     * @param string $event
     * @param array $arg
     * @return void
     */
    public function customerEvent($event, &$arg = null)
    {
        $customer_id = $this->customer->getId();

        if (!$this->dev5->status() || !$customer_id) {
            return;
        }

        if ($event !== 'edit') {
            return;
        }

        $custom_field = $this->dev5->db->querySelect(DB_PREFIX . 'customer', [
            'columns' => ['custom_field'],
            'where' => ['customer_id' => $customer_id]
        ]);

        if (!isset($custom_field[0]['custom_field'])) {
            return;
        }

        $custom_field = ocver('2.1', '<')
            ? unserialize($custom_field[0]['custom_field'])
            : json_decode($custom_field[0]['custom_field'], true);

        if (!$custom_field) {
            return;
        }

        !isset($arg['custom_field']) && $arg['custom_field'] = [];

        if (isset($custom_field['customer_d5bling_seller'])) {
            $arg['custom_field']['customer_d5bling_seller'] = $custom_field['customer_d5bling_seller'];
        }

        $this->request->post['custom_field'] = $arg['custom_field'];
    }

    /**
     * @return void
     */
    public function oauth2()
    {
        if (@$this->request->get['security'] !== $this->dev5->extension->security) {
            RWin::error('dev5-bling-oauth2', ['code' => 401, 'message' => 'Unauthorized']);
        }

        # Response
        if (!empty($this->request->get['state'])) {
            if (isset($this->request->request['error'])) {
                RWin::error('dev5-bling-oauth2', [
                    'code' => 406,
                    'message' => @$this->request->request['error_description'] ?: $this->request->request['error']
                ]);
                return;
            }

            if (empty($this->request->get['code'])) {
                RWin::error('dev5-bling-oauth2', ['code' => 406, 'message' => 'Bad Code']);
                return;
            }

            $state = explode(':', base64_decode($this->request->get['state']) ?: '');
            if (empty($state[0]) || empty($state[1])) {
                RWin::error('dev5-bling-oauth2', ['code' => 406, 'message' => 'Bad state']);
                return;
            }

            $api = new Api($state[0], $state[1]);
            if (!($access_token = $api->auth($this->request->get['code']))) {
                RWin::error('dev5-bling-oauth2', $api->error());
                return;
            }

            RWin::success('dev5-bling-oauth2', ['access_token' => $access_token]);
            return;
        }

        # Redirect
        if (isset($this->request->get['client_id'], $this->request->get['client_secret'])) {
            $api = new Api($this->request->get['client_id'], $this->request->get['client_secret']);
            $this->response->redirect($api->getAuthUrl(base64_encode("$api->client_id:$api->client_secret")));
            return;
        }

        RWin::error('dev5-bling-oauth2', ['code' => 400, 'message' => 'Bad Request']);
    }

    /**
     * @return void
     */
    public function callback()
    {
        if (@$this->request->get['security'] !== $this->dev5->extension->security) {
            http_response_code(401);
            exit;
        }

        if (!$this->dev5->status()) {
            http_response_code(503);
            exit;
        }

        if (!($callback = Api::getInstance()->getCallback())) {
            http_response_code(400);
            exit;
        }

        if (!in_array($callback['type'], ['order', 'stock'])) {
            http_response_code(406);
            exit;
        }

        set_time_limit(300);
        session_status() === PHP_SESSION_ACTIVE && session_write_close();

        global $callback_start_time;
        $callback_start_time = microtime(true);
        $this->dev5->debug('• CALLBACK-START');

        switch ($callback['type']) {
            case 'order':
                array_map(static function ($order) {
                    OrderEntity::getInstance()->importCallback($order, "|");
                }, $callback['data']);
                break;

            case 'stock':
                array_map(static function ($stock) {
                    ProductEntity::getInstance()->importCallback($stock, "|");
                }, $callback['data']);
                break;
        }

        $this->dev5->debug("|→ END", round(microtime(true) - $callback_start_time, 3) . 's');
        http_response_code(202);
        exit;
    }

    /**
     * @return void
     */
    public function cron()
    {
        if (@$this->request->get['security'] !== $this->dev5->extension->security) {
            http_response_code(401);
            exit;
        }

        if (!$this->dev5->status()) {
            http_response_code(503);
            exit;
        }

        global $cron_start_time;
        $cron_start_time = microtime(true);

        set_time_limit(300);
        session_status() === PHP_SESSION_ACTIVE && session_write_close();

        header('Content-Type: text/plain; charset=utf-8');
        $this->dev5->debug->setting([
            'mode' => $this->dev5->debug->setting()['mode'] === \Dev5\OpenCart\DEBUG_MODE_FILE
                ? \Dev5\OpenCart\DEBUG_MODE_ECHO_FILE
                : \Dev5\OpenCart\DEBUG_MODE_ECHO
        ]);
        $this->dev5->debug('• CRON-START');

        $start_time = microtime(true);
        do {
            if (connection_status() !== CONNECTION_NORMAL) {
                $this->dev5->debug("| • CONNECTION ABORTED");
                break;
            }

            $queue_start_time = microtime(true);

            if (!($entity = QueueEntity::consume())) {
                $this->dev5->debug("| • QUEUE EMPTY", "WAITING...");
                sleep(1);
                continue;
            }

            $this->dev5->debug("| • QUEUE", $entity->key);
            $payload = $entity->payload;

            list($type, $id) = explode(':', $entity->key);
            $payload['_type'] = $type;
            $payload['_id'] = $id;

            switch ($type) {
                case 'export_category':
                    $success = $payload['event'] === 'delete'
                        ? CategoryEntity::getInstance($id)->exportDeleteCategory($payload, "| |")
                        : CategoryEntity::getInstance($id)->exportCategory($payload, "| |");
                    break;

                case 'export_product':
                    $success = $payload['event'] === 'delete'
                        ? ProductEntity::getInstance($id)->exportDeleteProduct($payload, "| |")
                        : ProductEntity::getInstance($id)->exportProduct($payload, "| |");
                    break;

                case 'export_order':
                    $success = $payload['event'] === 'delete'
                        ? OrderEntity::getInstance($id)->exportDeleteOrder($payload, "| |")
                        : OrderEntity::getInstance($id)->exportOrder($payload, "| |");
                    break;

                default:
                    $success = false;
                    $error = ['code' => 406, 'message' => "Invalid queue type: $type"];
                    break;
            }

            if ($success) {
                $entity->status = QueueEntity::SUCCESS;
                $this->dev5->debug("| |→ SUCCESS", round(microtime(true) - $queue_start_time, 3) . 's');
            } else {
                $payload['error'] = !empty($error) ? $error : Api::getInstance()->error();
                $payload['debug'] = Api::getInstance()->last();
                $entity->status = QueueEntity::FAIL;
                $message = isset($payload['error']['message']) ? $payload['error']['message'] : '';
                $this->dev5->debug("| |→ FAIL", round(microtime(true) - $queue_start_time, 3) . 's,', $message); // phpcs:ignore
            }

            $payload['duration'] = microtime(true) - $queue_start_time;
            $entity->payload = $payload;
            $entity->save(['returning' => false]);
        } while (microtime(true) - $start_time < 60);

        $this->dev5->debug("• CRON-END", round(microtime(true) - $cron_start_time, 3) . 's');
    }
}

\Dev5\OpenCart::compile();
