<?php // phpcs:ignore

/**
 * This file is part of Dev5 Integração ERP Bling API V3 for OpenCart
 *
 * @author    Dev5™ <developer@dev5.com.br>
 * @copyright Dev5™ 2025. Todos os direitos reservados.
 * @license   Distribuição/reprodução não autorizada estará sujeita às penas das leis 9609/98 e 9610/98
 * @link      https://dev5.com.br/integracao-erp-bling-api-v3
 * @version   4.0.6
 */

namespace Dev5\OpenCart\Controller\Bling;

use Dev5\Contract\Database\Condition as C;
use Dev5\Library\Bling\Api;
use Dev5\Library\Bling\CategoryEntity;
use Dev5\Library\Bling\EIEntity;
use Dev5\Library\Bling\OrderEntity;
use Dev5\Library\Bling\ProductEntity;
use Dev5\Library\Bling\QueueEntity;
use Dev5\Library\RWin\RWin;

use function Dev5\Opencart\ocver;

abstract class Admin extends \Dev5\OpenCart
{
    /**
     * @return void
     */
    public function install()
    {
        CategoryEntity::migrate();
        ProductEntity::migrate();
        OrderEntity::migrate();
        EIEntity::migrate();
        QueueEntity::migrate();
    }

    /**
     * @return void
     */
    public function index()
    {
        $this->document->addScript($this->dev5->asset('rwin.min.js'));
        $this->response->setOutput($this->dev5->form([
            // phpcs:disable
            'bling' => [
                'type' => 'tabs',
                'label' => false,
                'tabs' => [
                    'general' => [
                        'fields' => [
                            'status' => 'onoff', // status
                            'auth_section' => [
                                'type' => 'section',
                                'fields' => [
                                    'auth_tutorial' => [
                                        'type' => 'html',
                                        'label' => true,
                                        'html' => $this->dev5->language->get('field_auth_tutorial_text', [
                                            'oauth2' => $this->dev5->link->catalog('oauth2', "security={$this->dev5->extension->security}")
                                        ])
                                    ],
                                    'client_id' => [ // client_id
                                        'type' => 'text',
                                        'invalidate' => 'value?.length !== 40' // required
                                    ],
                                    'client_secret' => [ // client_secret
                                        'type' => 'password',
                                        'invalidate' => 'value?.length !== 60' // required
                                    ],
                                    'access_token' => [ // access_token
                                        'type' => 'hidden',
                                        'invalidate' => true // required
                                    ],
                                    'error_token' => 'hidden', // error_token
                                    [
                                        'type' => 'html',
                                        'html' => function () {
                                            return $this->dev5->view([
                                                'bling_info' => Api::getInstance()->getDadosBasicos(['cache' => 10 * 60]),
                                                'href_oauth2' => $this->dev5->link->catalog('oauth2', "security={$this->dev5->extension->security}"),
                                                'href_clear_cache' => $this->dev5->link('clearCache'),
                                                'href_ei_event' => $this->dev5->link('eiEvent')
                                            ]);
                                        }
                                    ],
                                ]
                            ],
                            'callback_section' => [
                                'type' => 'section',
                                'fields' => [
                                    'callback_tutorial' => [
                                        'type' => 'html',
                                        'label' => true,
                                        'html' => $this->dev5->language->get('field_callback_tutorial_text', [
                                            'callback' => $this->dev5->link->catalog('callback', "security={$this->dev5->extension->security}")
                                        ])
                                    ],
                                    'channel_id' => [ // channel_id
                                        'type' => 'select',
                                        'options' => function () {
                                            return array_column(
                                                array_map(static function ($channel) {
                                                    return [
                                                        'id' => $channel['id'],
                                                        'label' => "#$channel[id] - $channel[descricao]"
                                                    ];
                                                }, array_filter(
                                                    Api::getInstance()->getCanaisVenda(['situacao' => 1], ['cache' => 7 * 24 * 60 * 60]),
                                                    static function ($channel) {
                                                        return in_array($channel['tipo'], ['Api', 'ApiDev5']);
                                                    }
                                                )),
                                                'label',
                                                'id'
                                            );
                                        },
                                        'invalidate' => true // required
                                    ]
                                ],
                                'invalidate' => '!$?.dev5?.form?.values()?.access_token'
                            ],
                            'cron_section' => [
                                'type' => 'section',
                                'fields' => [
                                    'cron_tutorial' => [
                                        'type' => 'html',
                                        'label' => true,
                                        'html' => $this->dev5->language->get('field_cron_tutorial_text', [
                                            'cron' => $cron = $this->dev5->link->catalog('cron', "security={$this->dev5->extension->security}")
                                        ])
                                    ],
                                    'command' => [
                                        'type' => 'input-group',
                                        'fields' => [
                                            '* * * * *',
                                            [
                                                'type' => 'html',
                                                'html' => "<span class='form-control'>$cron</span>"
                                            ]
                                        ]
                                    ]
                                ],
                                'invalidate' => '!$?.dev5?.form?.values()?.access_token'
                            ],
                            'others_section' => [
                                'type' => 'section',
                                'fields' => [
                                    'debug' => 'yesno',
                                    'image_url_tutorial' => [
                                        'type' => 'html',
                                        'label' => true,
                                        'html' => $this->dev5->language->get('field_image_url_tutorial_text')
                                    ],
                                    'cache_tutorial' => [
                                        'type' => 'html',
                                        'label' => true,
                                        'html' => $this->dev5->language->get('field_cache_tutorial_text')
                                    ]
                                ],
                                'invalidate' => '!$?.dev5?.form?.values()?.access_token'
                            ]
                        ],
                    ],
                    'product' => [
                        'fields' => [
                            'code_mode' => [ // code_mode
                                'type' => 'select',
                                'options' => [
                                    'product_id' => $this->dev5->language->get('field_sku_opt_product_id'),
                                    'sku' => $this->dev5->language->get('field_sku_opt_sku')
                                ],
                                'default' => 'product_id',
                                'invalidate' => true // required
                            ],
                            'view_mode' => [ // view_mode
                                'type' => 'select',
                                'options' => [
                                    'inline' => $this->dev5->language->get('field_view_mode_opt_inline'),
                                    'tab' => $this->dev5->language->get('field_view_mode_opt_tab')
                                ],
                                'default' => 'tab',
                                'invalidate' => true // required
                            ],
                            'language_id' => [ // language_id
                                'type' => 'select',
                                'options' => array_column(
                                    $opts = $this->model_localisation_language->getLanguages(),
                                    'name',
                                    'language_id'
                                ),
                                'default' => @array_find(
                                    $opts,
                                    static function ($language) {
                                        return stripos($language['code'], 'pt-br') !== false
                                            || stripos($language['name'], 'portugues') !== false
                                            || stripos($language['name'], 'português') !== false;
                                    }
                                )['language_id'],
                                'invalidate' => true // required
                            ],
                            'weight_class_id' => [ // weight_class_id
                                'type' => 'select',
                                'options' => array_column(
                                    $opts = $this->model_localisation_weight_class->getWeightClasses(),
                                    'title',
                                    'weight_class_id'
                                ),
                                'default' => @array_find(
                                    $opts,
                                    static function ($weight_class) {
                                        return stripos($weight_class['title'], 'kilo') !== false
                                            || stripos($weight_class['title'], 'quilo') !== false
                                            || stripos($weight_class['unit'], 'kg') !== false;
                                    }
                                )['weight_class_id'],
                                'invalidate' => true // required
                            ],
                            'length_class_id' => [ // length_class_id
                                'type' => 'select',
                                'options' => array_column(
                                    $opts = $this->model_localisation_length_class->getLengthClasses(),
                                    'title',
                                    'length_class_id'
                                ),
                                'default' => @array_find(
                                    $opts,
                                    static function ($length_class) {
                                        return stripos($length_class['title'], 'centimetro') !== false
                                            || stripos($length_class['title'], 'centímetro') !== false
                                            || stripos($length_class['unit'], 'cm') !== false;
                                    }
                                )['length_class_id'],
                                'invalidate' => true // required
                            ],
                            'currency_code' => [ // currency_code
                                'type' => 'select',
                                'options' => array_column(
                                    $this->model_localisation_currency->getCurrencies(),
                                    'title',
                                    'code'
                                ),
                                'default' => 'BRL',
                                'invalidate' => true // required
                            ],
                            'deposito' => [
                                'type' => 'select',
                                'options' => function () {
                                    return array_column(
                                        Api::getInstance()->getDepositos(['situacao' => 1], ['cache' => 7 * 24 * 60 * 60]),
                                        'descricao',
                                        'id'
                                    );
                                },
                                'invalidate' => true
                            ],
                            'canais' => [
                                'type' => 'well-checkbox',
                                'multiple' => true,
                                'options' => function () {
                                    return array_column(array_map(static function ($channel) {
                                        return [
                                            'value' => base64_encode(json_encode([
                                                'id' => $channel['id'],
                                                'descricao' => $channel['descricao']
                                            ], JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE)),
                                            'data' => [
                                                'label' => "#$channel[id] - $channel[descricao]",
                                                'attrs' => ["v-if" => "no_main_channel_id('$channel[id]')"]
                                            ]
                                        ];
                                    }, Api::getInstance()->getCanaisVenda(['situacao' => 1], ['cache' => 7 * 24 * 60 * 60])), 'data', 'value');
                                }
                            ]
                        ],
                        'invalidate' => '!$?.dev5?.form?.values()?.access_token'
                    ],
                    'order' => [
                        'fields' => [
                            'contato' => [
                                'type' => 'section',
                                'level' => true,
                                'fields' => [
                                    'codigo' => $this->getCustomerFields('/id do cliente/ui'), // contato[codigo]
                                    // contato[situacao], enum[A, I, E, S]
                                    'nome' => $this->getCustomerFields('/nome completo/ui'), // contato[nome]
                                    'fantasia' => $this->getCustomerFields('/(fantasia|nome completo)/ui'), // contato[fantasia]
                                    'email' => $this->getCustomerFields('/e-?mail/ui'), // contato[email]
                                    'telefone' => $this->getCustomerFields('/telefone/ui'), // contato[telefone]
                                    'celular' => $this->getCustomerFields('/celular/ui', false), // contato[celular]
                                    'cpf' => $this->getCustomerFields('/cpf/ui'), // contato[cpf]
                                    'cnpj' => $this->getCustomerFields('/cnpj/ui'), // contato[cnpj]
                                    'rg' => $this->getCustomerFields('/(rg|identidade)/ui', false), // contato[rg]
                                    'orgaoEmissor' => $this->getCustomerFields('/emissor/ui', false), // contato[orgaoEmissor]
                                    'ie' => $this->getCustomerFields('/(^ie$|inscri[cç][aã]o.*estadual)/ui', false), // contato[ie]
                                    'indicadorIe' => $this->getCustomerFields(false, false), // contato[indicadorIe]
                                    'defaultIndicadorIe' => [ // contato[defaultIndicadorIe]
                                        'type' => 'select',
                                        'options' => [
                                            '' => $this->dev5->language->get('field_contato_defaultIndicadorIe_opt_none'),
                                            '1' => $this->dev5->language->get('field_contato_defaultIndicadorIe_opt_1'),
                                            '2' => $this->dev5->language->get('field_contato_defaultIndicadorIe_opt_2'),
                                            '9' => $this->dev5->language->get('field_contato_defaultIndicadorIe_opt_9')
                                        ],
                                        'default' => '9'
                                    ],
                                    'im' => $this->getCustomerFields('/(^im$|inscri[cç][aã]o.*municipal)/ui', false), // contato[im]
                                    'dataNascimento' => $this->getCustomerFields('/nascimento/ui', false), // contato[dataNascimento]
                                    'sexo' => $this->getCustomerFields('/sexo/ui', false), // contato[sexo]
                                    'naturalidade' => $this->getCustomerFields('/naturalidade/ui', false), // contato[naturalidade]
                                    'endereco' => [
                                        'type' => 'level',
                                        'level' => true,
                                        'fields' => [
                                            'geral' => [
                                                'type' => 'section',
                                                'level' => true,
                                                'fields' => [
                                                    'nome' => $this->getCustomerFields('/entrega.*nome.*completo/ui'), // contato[endereco][geral][nome]
                                                    'cep' => $this->getCustomerFields('/entrega.*cep/ui'), // contato[endereco][geral][cep]
                                                    'endereco' => $this->getCustomerFields('/entrega.*endere[cç]o/ui'), // contato[endereco][geral][endereco]
                                                    'numero' => $this->getCustomerFields('/entrega.*n[uú]mero/ui'), // contato[endereco][geral][numero]
                                                    'bairro' => $this->getCustomerFields('/entrega.*bairro/ui'), // contato[endereco][geral][bairro]
                                                    'municipio' => $this->getCustomerFields('/entrega.*cidade/ui'), // contato[endereco][geral][municipio]
                                                    'uf' => $this->getCustomerFields('/entrega.*uf/ui'), // contato[endereco][geral][uf]
                                                    'complemento' => $this->getCustomerFields('/entrega.*complemento/ui', false), // contato[endereco][geral][complemento]
                                                ]
                                            ],
                                            'cobranca' => [
                                                'type' => 'section',
                                                'level' => true,
                                                'fields' => [
                                                    'nome' => $this->getCustomerFields('/fatura.*nome.*completo/ui'), // contato[fatura][geral][nome]
                                                    'cep' => $this->getCustomerFields('/fatura.*cep/ui'), // contato[endereco][cobranca][cep]
                                                    'endereco' => $this->getCustomerFields('/fatura.*endere[cç]o/ui'), // contato[endereco][cobranca][endereco]
                                                    'numero' => $this->getCustomerFields('/fatura.*n[uú]mero/ui'), // contato[endereco][cobranca][numero]
                                                    'bairro' => $this->getCustomerFields('/fatura.*bairro/ui'), // contato[endereco][cobranca][bairro]
                                                    'municipio' => $this->getCustomerFields('/fatura.*cidade/ui'), // contato[endereco][cobranca][municipio]
                                                    'uf' => $this->getCustomerFields('/fatura.*uf/ui'), // contato[endereco][cobranca][uf]
                                                    'complemento' => $this->getCustomerFields('/fatura.*complemento/ui', false), // contato[endereco][cobranca][complemento]
                                                ]
                                            ],
                                        ]
                                    ],
                                    'campos_customizados' => [
                                        'type' => 'section',
                                        'level' => true,
                                        'fields' => function () {
                                            return array_column(array_filter(array_map(function ($i) {
                                                if ($i['situacao'] !== 1) {
                                                    return false;
                                                }

                                                $this->dev5->language->set("field_contato_campos_customizados_$i[id]", $i['nome']);
                                                $this->dev5->language->set("field_contato_campos_customizados_$i[id]_help", "<p>{$this->dev5->language->get('text_type')}: {$i['tipoCampo']['nome']}</p>$i[placeholder]");
                                                $this->dev5->language->set("field_contato_campos_customizados_$i[id]_invalid", $this->dev5->language->get('field_invalid_required'));

                                                return ['campos_customizados_id' => $i['id'], 'invalidate' => !!$i['obrigatorio']]
                                                    + $this->getCustomerFields(false, false);
                                            }, Api::getInstance()->getCamposCustomizadosModulo('Contatos', ['cache' => 7 * 24 * 60 * 60]))), null, 'campos_customizados_id') ?: [
                                                [
                                                    'type' => 'html',
                                                    'html' => "<h4 class=\"text-center\">{$this->dev5->language->get('field_contato_campos_customizados_none')}</h4>"
                                                ]
                                            ];
                                        }
                                    ]
                                ]
                            ],
                            'order_status' => [
                                'type' => 'section',
                                'level' => true,
                                'fields' => function () {
                                    return array_column(
                                        array_map(
                                            function ($status) {
                                                $this->dev5->language->set("field_order_status_$status[id]", "$status[nome] <i class=\"fa fa-circle\" style=\"color:$status[cor]\"></i>");
                                                $this->dev5->language->set("field_order_status_$status[id]_help", $this->dev5->language->get("field_order_status_help"));
                                                return [
                                                    'bling_status_id' => $status['id'],
                                                    'type' => 'input-group',
                                                    'level' => true,
                                                    'fields' => [
                                                        'status_id' => [
                                                            'type' => 'select',
                                                            'options' => array_column(
                                                                array_merge(
                                                                    [['order_status_id' => '', 'name' => $this->dev5->language->get('field_order_status_ignore')]],
                                                                    $this->model_localisation_order_status->getOrderStatuses()
                                                                ),
                                                                'name',
                                                                'order_status_id'
                                                            )
                                                        ],
                                                        'stock' => [
                                                            'type' => 'select',
                                                            'options' => [
                                                                '' => $this->dev5->language->get('field_order_status_stock_opt_none'),
                                                                '1' => $this->dev5->language->get('field_order_status_stock_opt_reduce'),
                                                                '2' => $this->dev5->language->get('field_order_status_stock_opt_reverse')
                                                            ],
                                                            'default' => ''
                                                        ],
                                                        'notify' => [
                                                            'type' => 'checkbox',
                                                            'options' => ['1' => $this->dev5->language->get('field_order_status_notify')],
                                                            'default' => '1',
                                                            'multiple' => false
                                                        ]
                                                    ]
                                                ];
                                            },
                                            Api::getInstance()->getSituacoesModulo('Vendas', ['cache' => 7 * 24 * 60 * 60])
                                        ),
                                        null,
                                        'bling_status_id'
                                    ) ?: [[
                                        'type' => 'html',
                                        'html' => "<h4 class=\"text-center\">{$this->dev5->language->get('field_order_status_none')}</h4>"
                                    ]];
                                }
                            ],
                            'payment_method' => [
                                'type' => 'section',
                                'level' => true,
                                'fields' => function () {
                                    return array_column(
                                        array_map(
                                            function ($payment) {
                                                $this->dev5->language->set("field_payment_method_$payment[id]", $payment['descricao']);
                                                $this->dev5->language->set("field_payment_method_$payment[id]_help", $this->dev5->language->get("field_payment_method_help"));
                                                return [
                                                    'bling_payment_code' => $payment['id'],
                                                    'type' => 'input-group',
                                                    'fields' => [
                                                        $payment['id'] => [
                                                            'type' => 'autocomplete',
                                                            'callback' => function ($value) {
                                                                return $this->autocompletePaymentMethods($value);
                                                            },
                                                            'required' => false,
                                                            'invalidate' => "(!value?.value || /^\w+$/.test(value.value)) ? !1 : '{$this->dev5->language->get('field_payment_method_invalid_code_format')}'",
                                                            "attrs" => ['placeholder' => $this->dev5->language->get('field_payment_method_placeholder')],
                                                        ],
                                                        'days' => [
                                                            'type' => 'number',
                                                            'level' => ['payment_method', $payment['id']],
                                                            'default' => '',
                                                            'attrs' => [
                                                                'dev5-tooltip' => $this->dev5->language->get('field_payment_method_days_help'),
                                                                'placeholder' => $this->dev5->language->get('field_payment_method_days_placeholder')
                                                            ]
                                                        ],
                                                        'regex' => [
                                                            'type' => 'checkbox',
                                                            'level' => ['payment_method', $payment['id']],
                                                            'options' => ['1' => $this->dev5->language->get('field_payment_method_regex')],
                                                            'default' => '1',
                                                            'multiple' => false
                                                        ]
                                                    ]
                                                ];
                                            },
                                            Api::getInstance()->getFormasDePagamentos(['situacao' => 1], ['cache' => 7 * 24 * 60 * 60])
                                        ),
                                        null,
                                        'bling_payment_code'
                                    ) ?: [[
                                        'type' => 'html',
                                        'html' => "<h4 class=\"text-center\">{$this->dev5->language->get('field_payment_method_none')}</h4>"
                                    ]];
                                }
                            ],
                            'shipping_method' => [
                                'type' => 'section',
                                'level' => true,
                                'fields' => function () {
                                    return array_column(
                                        array_merge([], ...array_filter(array_map(
                                            function ($shipping) {
                                                if (empty($shipping['servicos'])) {
                                                    $this->dev5->language->set("field_shipping_method_$shipping[id]", $shipping['descricao']);
                                                    $this->dev5->language->set("field_shipping_method_$shipping[id]_help", $this->dev5->language->get("field_shipping_method_help"));

                                                    return [
                                                        [
                                                            'bling_shipping_code' => $shipping['id'],
                                                            'type' => 'input-group',
                                                            'fields' => [
                                                                $shipping['id'] => [
                                                                    'type' => 'autocomplete',
                                                                    'callback' => function ($value) {
                                                                        return $this->autocompleteShippingMethods($value);
                                                                    },
                                                                    'required' => false,
                                                                    'invalidate' => "(!value?.value || /^\w+\.(\w+|\*)$/.test(value.value)) ? !1 : '{$this->dev5->language->get('field_shipping_method_invalid_code_format')}'",
                                                                    "attrs" => ['placeholder' => $this->dev5->language->get('field_shipping_method_placeholder')]
                                                                ],
                                                                'fretePorConta' => [
                                                                    'type' => 'select',
                                                                    'level' => ['shipping_method', $shipping['id']],
                                                                    'options' => [
                                                                        '' => $this->dev5->language->get('field_shipping_method_fretePorConta_opt_0'),
                                                                        '1' => $this->dev5->language->get('field_shipping_method_fretePorConta_opt_1'),
                                                                        '2' => $this->dev5->language->get('field_shipping_method_fretePorConta_opt_2'),
                                                                        '3' => $this->dev5->language->get('field_shipping_method_fretePorConta_opt_3'),
                                                                        '4' => $this->dev5->language->get('field_shipping_method_fretePorConta_opt_4'),
                                                                        '9' => $this->dev5->language->get('field_shipping_method_fretePorConta_opt_9')
                                                                    ],
                                                                    'default' => '',
                                                                    'attrs' => [
                                                                        'dev5-tooltip' => $this->dev5->language->get('field_shipping_method_fretePorConta_help')
                                                                    ]
                                                                ],
                                                                'logistic' => [
                                                                    'type' => 'hidden',
                                                                    'level' => ['shipping_method', $shipping['id']],
                                                                    'value' => $shipping['descricao']
                                                                ],
                                                                'service' => [
                                                                    'type' => 'hidden',
                                                                    'level' => ['shipping_method', $shipping['id']],
                                                                    'value' => ''
                                                                ]
                                                            ]
                                                        ]
                                                    ];
                                                }

                                                $shipping = Api::getInstance()->getLogistica($shipping['id'], ['cache' => 7 * 24 * 60 * 60]);
                                                return array_map(function ($service) use ($shipping) {
                                                    $this->dev5->language->set("field_shipping_method_$shipping[id]_$service[id]", "$shipping[descricao] - $service[descricao]");
                                                    $this->dev5->language->set("field_shipping_method_$shipping[id]_$service[id]_help", $this->dev5->language->get("field_shipping_method_help"));
                                                    return [
                                                        'bling_shipping_code' => "$shipping[id]_$service[id]",
                                                        'type' => 'input-group',
                                                        'fields' => [
                                                            "$shipping[id]_$service[id]" => [
                                                                'type' => 'autocomplete',
                                                                'callback' => function ($value) {
                                                                    return $this->autocompleteShippingMethods($value);
                                                                },
                                                                'required' => false,
                                                                'invalidate' => "(!value?.value || /^\w+\.(\w+|\*)$/.test(value.value)) ? !1 : '{$this->dev5->language->get('field_shipping_method_invalid_code_format')}'",
                                                                "attrs" => ['placeholder' => $this->dev5->language->get('field_shipping_method_placeholder')]
                                                            ],
                                                            'fretePorConta' => [
                                                                'type' => 'select',
                                                                'level' => ['shipping_method', "$shipping[id]_$service[id]"],
                                                                'options' => [
                                                                    '' => $this->dev5->language->get('field_shipping_method_fretePorConta_opt_0'),
                                                                    '1' => $this->dev5->language->get('field_shipping_method_fretePorConta_opt_1'),
                                                                    '2' => $this->dev5->language->get('field_shipping_method_fretePorConta_opt_2'),
                                                                    '3' => $this->dev5->language->get('field_shipping_method_fretePorConta_opt_3'),
                                                                    '4' => $this->dev5->language->get('field_shipping_method_fretePorConta_opt_4'),
                                                                    '9' => $this->dev5->language->get('field_shipping_method_fretePorConta_opt_9')
                                                                ],
                                                                'default' => '',
                                                                'attrs' => [
                                                                    'dev5-tooltip' => $this->dev5->language->get('field_shipping_method_fretePorConta_help')
                                                                ]
                                                            ],
                                                            'logistic' => [
                                                                'type' => 'hidden',
                                                                'level' => ['shipping_method', "$shipping[id]_$service[id]"],
                                                                'value' => $shipping['descricao']
                                                            ],
                                                            'service' => [
                                                                'type' => 'hidden',
                                                                'level' => ['shipping_method', "$shipping[id]_$service[id]"],
                                                                'value' => !empty($service['aliases']) ? array_shift($service['aliases']) : $service['descricao']
                                                            ]
                                                        ]
                                                    ];
                                                }, $shipping['servicos']);
                                            },
                                            Api::getInstance()->getLogisticas(['situacao' => 'H'], ['cache' => 7 * 24 * 60 * 60])
                                        ))),
                                        null,
                                        'bling_shipping_code'
                                    ) ?: [[
                                        'type' => 'html',
                                        'html' => "<h4 class=\"text-center\">{$this->dev5->language->get('field_shipping_method_none')}</h4>"
                                    ]];
                                }
                            ]
                        ],
                        'invalidate' => '!$?.dev5?.form?.values()?.access_token'
                    ],
                    'ei' => [
                        'fields' => [
                            [
                                "type" => "html",
                                "html" => "<div class=\"alert alert-info\">{$this->dev5->language->get("field_ei_alert")}</div>"
                            ],
                            "category" => [
                                "type" => "section",
                                "level" => ['ei'],
                                "fields" => [
                                    [
                                        "type" => "input-group",
                                        "label" => false,
                                        "fields" => [
                                            "export_category" => [
                                                "type" => "well-checkbox",
                                                "options" => [
                                                    'add' => $this->dev5->language->get("field_ei_export_category_opt_add"),
                                                    'edit' => $this->dev5->language->get("field_ei_export_category_opt_edit"),
                                                    'delete' => $this->dev5->language->get("field_ei_export_category_opt_delete")
                                                ],
                                                'default' => ['add', 'edit', 'delete']
                                            ],
                                            ["type" => "html", "html" => $this->dev5->language->get("field_ei_export_category")]
                                        ]
                                    ]
                                ]
                            ],
                            "product" => [
                                "type" => "section",
                                "level" => ['ei'],
                                "fields" => [
                                    [
                                        "type" => "input-group",
                                        "label" => false,
                                        "fields" => [
                                            "export_product" => [
                                                "type" => "well-checkbox",
                                                "options" => [
                                                    'add' => $this->dev5->language->get("field_ei_export_product_opt_add"),
                                                    'edit' => $this->dev5->language->get("field_ei_export_product_opt_edit"),
                                                    'delete' => $this->dev5->language->get("field_ei_export_product_opt_delete")
                                                ],
                                                'default' => ['add', 'edit', 'delete']
                                            ],
                                            ["type" => "html", "html" => $this->dev5->language->get("field_ei_export_product")]
                                        ]
                                    ],
                                    [
                                        "type" => "input-group",
                                        "label" => false,
                                        "fields" => [
                                            "import_stock_callback" => [
                                                "type" => "onoff",
                                                'default' => '1'
                                            ],
                                            ["type" => "html", "html" => $this->dev5->language->get("field_ei_import_stock_callback")]
                                        ]
                                    ],
                                    [
                                        "type" => "input-group",
                                        "label" => false,
                                        "fields" => [
                                            "export_product_with_category" => [
                                                "type" => "onoff",
                                                'default' => '1'
                                            ],
                                            ["type" => "html", "html" => $this->dev5->language->get("field_ei_export_product_with_category")]
                                        ]
                                    ],
                                    [
                                        "type" => "input-group",
                                        "label" => false,
                                        "fields" => [
                                            "export_product_with_stock" => [
                                                "type" => "onoff",
                                                'default' => '1'
                                            ],
                                            ["type" => "html", "html" => $this->dev5->language->get("field_ei_export_product_with_stock")]
                                        ]
                                    ]
                                ]
                            ],
                            "order" => [
                                "type" => "section",
                                "level" => ['ei'],
                                "fields" => [
                                    [
                                        "type" => "input-group",
                                        "label" => false,
                                        "fields" => [
                                            "export_order" => [
                                                "type" => "well-checkbox",
                                                "options" => [
                                                    'add' => $this->dev5->language->get("field_ei_export_order_opt_add"),
                                                    'edit' => $this->dev5->language->get("field_ei_export_order_opt_edit"),
                                                    'delete' => $this->dev5->language->get("field_ei_export_order_opt_delete"),
                                                    'status' => $this->dev5->language->get("field_ei_export_order_opt_status"),
                                                ],
                                                'default' => ['add', 'edit', 'delete', 'status']
                                            ],
                                            ["type" => "html", "html" => $this->dev5->language->get("field_ei_export_order")]
                                        ]
                                    ],
                                    [
                                        "type" => "input-group",
                                        "label" => false,
                                        "fields" => [
                                            "import_order_callback" => [
                                                "type" => "onoff",
                                                'default' => '1'
                                            ],
                                            ["type" => "html", "html" => $this->dev5->language->get("field_ei_import_order_callback")]
                                        ]
                                    ],
                                    [
                                        "type" => "input-group",
                                        "label" => false,
                                        "fields" => [
                                            "export_order_ignore_id" => "number",
                                            ["type" => "html", "html" => $this->dev5->language->get("field_ei_export_order_ignore_id")]
                                        ]
                                    ],
                                    [
                                        "type" => "input-group",
                                        "label" => false,
                                        "fields" => [
                                            "export_order_ignore_date" => "date",
                                            ["type" => "html", "html" => $this->dev5->language->get("field_ei_export_order_ignore_date")]
                                        ]
                                    ],
                                    [
                                        "type" => "input-group",
                                        "label" => false,
                                        "fields" => [
                                            "import_order_not_found" => [
                                                "type" => "select",
                                                "options" => [
                                                    "create" => $this->dev5->language->get("field_ei_import_order_not_found_opt_create"),
                                                    "ignore" => $this->dev5->language->get("field_ei_import_order_not_found_opt_ignore")
                                                ],
                                                'default' => 'create'
                                            ],
                                            ["type" => "html", "html" => $this->dev5->language->get("field_ei_import_order_not_found")]
                                        ]
                                    ],
                                    [
                                        "type" => "input-group",
                                        "label" => false,
                                        "fields" => [
                                            "update_seller_on_event" => [
                                                "type" => "select",
                                                'options' => [
                                                    '' => $this->dev5->language->get("field_ei_update_seller_on_event_opt_none"),
                                                    'last' => $this->dev5->language->get("field_ei_update_seller_on_event_opt_last"),
                                                    'current' => $this->dev5->language->get("field_ei_update_seller_on_event_opt_current")
                                                ],
                                                'default' => 'last'
                                            ],
                                            ["type" => "html", "html" => $this->dev5->language->get("field_ei_update_seller_on_event")]
                                        ]
                                    ]
                                ]
                            ]
                        ],
                        'invalidate' => '!$?.dev5?.form?.values()?.access_token'
                    ],
                    'queue' => [
                        'align' => 'right',
                        'fields' => [
                            'queue' => [
                                'type' => 'datatable',
                                'label' => false,
                                'columns' => [
                                    'timestamp' => [
                                        'orderable' => false,
                                        'render' => "
                                            return moment(row.timestamp).format('DD/MM/YYYY HH:mm:ss');
                                        "
                                    ],
                                    'event' => [
                                        'orderable' => false,
                                        'render' => "
                                            let [event, id] = row.key.split(':');
                                            switch (event) {
                                                case 'export_category':
                                                    return `{$this->dev5->language->get('text_queue_export_category')} #\${id}`;
                                                case 'delete_category':
                                                    return `{$this->dev5->language->get('text_queue_delete_category')} #\${id}`;

                                                case 'export_product':
                                                    return `{$this->dev5->language->get('text_queue_export_product')} #\${id}`;
                                                case 'delete_product':
                                                    return `{$this->dev5->language->get('text_queue_delete_product')} #\${id}`;

                                                case 'export_order':
                                                    return `{$this->dev5->language->get('text_queue_export_order')} #\${id}`;
                                                case 'delete_order':
                                                    return `{$this->dev5->language->get('text_queue_delete_order')} #\${id}`;

                                                case 'import_category':
                                                    return `{$this->dev5->language->get('text_queue_import_category')} #\${id}`;

                                                case 'import_product':
                                                    return `{$this->dev5->language->get('text_queue_import_product')} #\${id}`;

                                                case 'import_order':
                                                    return `{$this->dev5->language->get('text_queue_import_order')} #\${id}`;

                                                case 'callback_order_status':
                                                    return `{$this->dev5->language->get('text_queue_callback_order_status')} #\${id}`;

                                                case 'callback_product_stock':
                                                    return `{$this->dev5->language->get('text_queue_callback_product_stock')} #\${id}`;

                                                default:
                                                    return `UNKNOWN[\${event}] #\${id}`;
                                            }
                                        "
                                    ],
                                    'status' => [
                                        'align' => 'right',
                                        'orderable' => false,
                                        'render' => "
                                            switch (row.status) {
                                                case " . QueueEntity::PENDING . ":
                                                    return `<span class=\"label label-default\">{$this->dev5->language->get('text_queue_pending')}</span>`;

                                                case " . QueueEntity::PROCESSING . ":
                                                    return `<span class=\"label label-primary\">{$this->dev5->language->get('text_queue_processing')}</span>`;

                                                case " . QueueEntity::SUCCESS . ":
                                                    return `<span class=\"label label-success\">{$this->dev5->language->get('text_queue_success')}</span>`;

                                                case " . QueueEntity::FAIL . ":
                                                    return `<span class=\"label label-danger\" dev5-tooltip=\"\${row.error}\">{$this->dev5->language->get('text_queue_error')}</span>`;

                                                default:
                                                    return `<span class=\"label label-warning\">UNKNOWN[\${row.status}]</span>`;
                                            }
                                        "
                                    ]
                                ],
                                'callback' => function ($filters, $order, $start, $length) {
                                    return [
                                        'recordsTotal' => $recordsTotal = QueueEntity::findTotal(),
                                        'recordsFiltered' => $recordsTotal,
                                        'data' => array_map(function ($row) {
                                            $row['status'] = intval($row['status']);
                                            $row['payload'] = json_decode($row['payload'], true);

                                            $error = null;

                                            if ($row['status'] === QueueEntity::FAIL) {
                                                $error = $this->dev5->language->get('error_unknown');
                                                if (!empty($row['payload']['error']['message'])) {
                                                    # Internal Error
                                                    $error = $row['payload']['error']['message'];
                                                } elseif (!empty($row['payload']['error'])) {
                                                    # Internal Error
                                                    $error = $row['payload']['error'];
                                                }

                                                if (!empty($row['payload']['debug']['response'])) {
                                                    $debug = $row['payload']['debug']['response'];

                                                    if (!empty($debug['content']['error'])) {
                                                        # Bling Error
                                                        $debug = $debug['content']['error'];

                                                        if (!empty($debug['description'])) {
                                                            $error = $debug['description'];
                                                        } elseif (!empty($debug['message'])) {
                                                            $error = $debug['message'];
                                                        }

                                                        if (!empty($debug['fields'])) {
                                                            $error = "<p>$error</p>";

                                                            foreach ($debug['fields'] as $field) {
                                                                $error .= PHP_EOL . "<p>#$field[code] $field[msg]</p>";
                                                            }
                                                        }
                                                    } elseif (!empty($debug['errno'])) {
                                                        # cUrl Error
                                                        $error = "#$debug[errno] $debug[error]";
                                                    }
                                                }
                                            }

                                            switch ($row['key']) {
                                                case 'export_category':
                                                case 'export_product':
                                                case 'export_order':
                                                    $row['payload']['event'] === 'delete'
                                                    && $row['key'] = str_replace('export_', 'delete_', $row['key']);
                                                    break;
                                            }

                                            if (is_array($error)) {
                                                $error = json_encode($error, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE); // phpcs:ignore
                                            }

                                            return [
                                                'key' => $row['key'],
                                                'status' => $row['status'],
                                                'error' => htmlentities(nl2br($error ?: '')),
                                                'debug' => $row,
                                                'timestamp' => $row['timestamp']
                                            ];
                                        }, $this->dev5->db->query("
                                            SELECT
                                                `key`,
                                                `status`,
                                                `payload`,
                                                `timestamp`
                                            FROM
                                                `" . QueueEntity::proxy()->table->name . "`
                                            ORDER BY
                                                `timestamp` DESC,
                                                `status` = '" . QueueEntity::PROCESSING . "' DESC
                                            LIMIT
                                                $start, $length
                                        ")->all())
                                    ];
                                },
                            ]
                        ],
                        'invalidate' => '!$?.dev5?.form?.values()?.access_token'
                    ]
                ]
            ]
            // phpcs:enable
        ]));
    }

    /**
     * @return void
     */
    public function clearCache()
    {
        Api::getInstance()->clearCache();
        $this->session->data["success.{$this->dev5->extension->id}"] = $this->dev5->language['text_success_clear_cache']; // phpcs:ignore
        $this->response->redirect($this->dev5->link->extension);
    }

    /**
     * @param array $data
     * @return void
     */
    public function categoryForm(&$data = [])
    {
        if (!$this->dev5->status()) {
            return;
        }

        $data['d5bling'] = $this->dev5->view([
            'flag' => $this->dev5->asset('bling/logo.webp'),
            'language_id' => $this->dev5->field('language_id')
        ], $this->dev5->route() . "_category_form");
    }

    /**
     * @param array $error
     * @return void
     */
    public function categoryValidate(&$error = [])
    {
        // nothing to do
    }

    /**
     * @param string $event
     * @param int $category_id
     * @return void
     */
    public function categoryEvent($event, $category_id = null)
    {
        if (CategoryEntity::ignore()) {
            return;
        }

        if (!$this->dev5->status()) {
            return;
        }

        if (empty($category_id) || !is_numeric($category_id)) {
            return;
        }

        if (!in_array($event, @$this->dev5->field('ei')['export_category'] ?: [])) {
            return;
        }

        QueueEntity::publish("export_category:$category_id", ['event' => $event]);
    }

    /**
     * @param array $data
     * @return void
     */
    public function productForm(&$data = [])
    {
        if (!$this->dev5->status()) {
            return;
        }

        $product_id = isset($this->request->get['product_id']) ? $this->request->get['product_id'] : 0;

        if ($this->request->server['REQUEST_METHOD'] === 'POST') {
            $values = @$this->request->post['product_d5bling'] ?: [];
            $this->productValidate($error);
        } elseif ($entity = ProductEntity::findOne(['product_id' => $product_id, 'options' => ''])) {
            $values = $entity->toArray();
            $values['data_validade'] = $entity->data_validade ? $entity->data_validade->format('Y-m-d') : '';
            $error = null;
        } else {
            $values = [];
            $error = null;
        }

        $data['d5bling'] = [
            'tab' => "
                <li>
                    <a href=\"#tab-d5bling\" data-toggle=\"tab\" data-bs-toggle=\"tab\">
                        <img src=\"{$this->dev5->asset('bling/logo.webp')}\" alt=\"Bling\">
                    </a>
                </li>
            ",
            'content' =>
                '<div class="tab-pane" id="tab-d5bling">'
                . $this->dev5->form([
                    // phpcs:disable
                    'product_d5bling' => [
                        'type' => 'level',
                        'level' => true,
                        'fields' => [
                            // id, int
                            // name, string
                            // codigo, string
                            // preco, float
                            'canais' => [
                                'type' => 'table-columns',
                                'action' => false,
                                'fields' => [
                                    'channel_id' => 'hidden', // canais[*][channel_id]
                                    'channel_status' => [ // canais[*][channel_status]
                                        'type' => 'checkbox',
                                        'options' => ['1' => ''],
                                        'multiple' => false
                                    ],
                                    'channel_name' => 'html',
                                    'additional_price' => [
                                        'type' => 'input-group',
                                        'level' => true,
                                        'fields' => [
                                            'type' => [ // canais[*][additional_price][type]
                                                'type' => 'select',
                                                'options' => ['' => 'R$', 'P' => '%'],
                                                'attrs' => ['style' => 'width:75px']
                                            ],
                                            'value' => [ // canais[*][additional_price][value]
                                                'type' => 'number',
                                                'attrs' => ['placeholder' => '0']
                                            ]
                                        ]
                                    ],
                                    'additional_special' => [
                                        'type' => 'input-group',
                                        'level' => true,
                                        'fields' => [
                                            'type' => [ // canais[*][additional_special][type]
                                                'type' => 'select',
                                                'options' => ['' => 'R$', 'P' => '%'],
                                                'attrs' => ['style' => 'width:75px']
                                            ],
                                            'value' => [ // canais[*][additional_special][value]
                                                'type' => 'number',
                                                'attrs' => ['placeholder' => '0']
                                            ]
                                        ]
                                    ]
                                ],
                                'value' => array_map(static function ($channel) use ($values) {
                                    $channel = json_decode(base64_decode($channel), true);

                                    $value = array_find(
                                        @$values['canais'] ?: [],
                                        static function ($i) use ($channel) {
                                            return @$i['channel_id'] == @$channel['id'];
                                        }
                                    ) ?: [];

                                    $value['channel_id'] = $channel['id'];
                                    $value['channel_name'] = "#$channel[id] - $channel[descricao]";
                                    $value['channel_status'] = !!@$value['channel_status'];
                                    $value['additional_price'] = @$value['additional_price'] ?: [];
                                    $value['additional_price']['type'] = @$value['additional_price']['type'] ?: '';
                                    $value['additional_price']['value'] = @$value['additional_price']['value'] ?: '';
                                    $value['additional_special'] = @$value['additional_special'] ?: [];
                                    $value['additional_special']['type'] = @$value['additional_special']['type'] ?: '';
                                    $value['additional_special']['value'] = @$value['additional_special']['value'] ?: '';
                                    return $value;
                                }, $this->dev5->field('canais') ?: [])
                            ],
                            'tipo' => [ // d5bling[tipo]
                                'type' => 'select',
                                'options' => [
                                    'P' => $this->dev5->language->get('field_product_form_tipo_opt_P'),
                                    'S' => $this->dev5->language->get('field_product_form_tipo_opt_S')
                                ],
                                'default' => 'P'
                            ],
                            // situacao, enum[A, I]
                            // formato, enum[S, V, E]
                            // descricaoCurta
                            'data_validade' => 'date', // d5bling[data_validade]
                            'unidade' => [ // d5bling[unidade]
                                'type' => 'datalist',
                                'options' => [
                                    '' => $this->dev5->language->get('field_product_d5bling_unidade_opt_none'),
                                    'UN' => $this->dev5->language->get('field_product_d5bling_unidade_opt_UN'),
                                    'PC' => $this->dev5->language->get('field_product_d5bling_unidade_opt_PC'),
                                    'CX' => $this->dev5->language->get('field_product_d5bling_unidade_opt_CX'),
                                    'KG' => $this->dev5->language->get('field_product_d5bling_unidade_opt_KG'),
                                    'M' => $this->dev5->language->get('field_product_d5bling_unidade_opt_M'),
                                    'L' => $this->dev5->language->get('field_product_d5bling_unidade_opt_L'),
                                ],
                                'default' => 'UN'
                            ],
                            'deposito' => [
                                'type' => 'select',
                                'options' => function () {
                                    return array_column(
                                        Api::getInstance()->getDepositos(['situacao' => 1], ['cache' => 7 * 24 * 60 * 60]),
                                        'descricao',
                                        'id'
                                    );
                                },
                                'invalidate' => true
                            ],
                            // pesoBruto, float(kg)
                            'peso_liquido' => 'number', // d5bling[peso_liquido]
                            'volumes' => 'number', // d5bling[volumes]
                            'itens_por_caixa' => 'number', // d5bling[itens_por_caixa]
                            'gtin' => 'text', // d5bling[gtin]
                            'gtin_embalagem' => 'text', // d5bling[gtin_embalagem]
                            'tipo_producao' => [ // d5bling[tipo_producao]
                                'type' => 'select',
                                'options' => [
                                    'P' => $this->dev5->language->get('field_product_form_tipo_producao_opt_P'),
                                    'T' => $this->dev5->language->get('field_product_form_tipo_producao_opt_T')
                                ]
                            ],
                            'condicao' => [ // d5bling[condicao]
                                'type' => 'select',
                                'options' => [
                                    0 => $this->dev5->language->get('field_product_form_condicao_opt_0'),
                                    1 => $this->dev5->language->get('field_product_form_condicao_opt_1'),
                                    2 => $this->dev5->language->get('field_product_form_condicao_opt_2')
                                ]
                            ],
                            'frete_gratis' => 'yesno', // d5bling[frete_gratis]
                            // marca, string
                            // descricaoComplementar, string
                            // linkExterno, string
                            'observacoes' => 'textarea', // d5bling[observacoes]
                            'descricao_embalagem_discreta' => 'textarea', // d5bling[descricao_embalagem_discreta]
                            'categoria' => [
                                'type' => 'level',
                                'level' => true,
                                'fields' => [
                                    'id' => [ // d5bling[categoria][id]
                                        'type' => 'autocomplete',
                                        'callback' => function ($value) {
                                            return $this->autocompleteCategories($value);
                                        }
                                    ],
                                ]
                            ],
                            'estoque' => [
                                'type' => 'level',
                                'level' => true,
                                'fields' => [
                                    'minimo' => 'number', // d5bling[estoque][minimo]
                                    'maximo' => 'number', // d5bling[estoque][maximo]
                                    'crossdocking' => 'number', // d5bling[estoque][crossdocking]
                                    // estoque[localizacao], string
                                ]
                            ],
                            // actionEstoque, enum[Z, T]
                            // dimensoes[largura], float
                            // dimensoes[altura], float
                            // dimensoes[profundidade], float
                            // dimensoes[unidadeMedida], enum[0, 1, 2]
                            'tributacao' => [
                                'type' => 'level',
                                'level' => true,
                                'fields' => [
                                    'origem' => [ // d5bling[tributacao][origem]
                                        'type' => 'select',
                                        'options' => [
                                            '' => $this->dev5->language->get('field_product_d5bling_tributacao_origem_opt_none'),
                                            0 => $this->dev5->language->get('field_product_d5bling_tributacao_origem_opt_0'),
                                            1 => $this->dev5->language->get('field_product_d5bling_tributacao_origem_opt_1'),
                                            2 => $this->dev5->language->get('field_product_d5bling_tributacao_origem_opt_2'),
                                            3 => $this->dev5->language->get('field_product_d5bling_tributacao_origem_opt_3'),
                                            4 => $this->dev5->language->get('field_product_d5bling_tributacao_origem_opt_4'),
                                            5 => $this->dev5->language->get('field_product_d5bling_tributacao_origem_opt_5'),
                                            6 => $this->dev5->language->get('field_product_d5bling_tributacao_origem_opt_6'),
                                            7 => $this->dev5->language->get('field_product_d5bling_tributacao_origem_opt_7'),
                                            8 => $this->dev5->language->get('field_product_d5bling_tributacao_origem_opt_8')
                                        ]
                                    ],
                                    'nFCI' => 'text', // d5bling[tributacao][nFCI]
                                    'ncm' => 'text', // d5bling[tributacao][ncm]
                                    'cest' => 'text', // d5bling[tributacao][cest]
                                    'codigoListaServicos' => [ // d5bling[tributacao][codigoListaServicos]
                                        'type' => 'text',
                                        'mask' => '99.99'
                                    ],
                                    'spedTipoItem' => [ // d5bling[tributacao][origem]
                                        'type' => 'select',
                                        'options' => [
                                            '' => $this->dev5->language->get('field_product_d5bling_tributacao_spedTipoItem_help'),
                                            '00' => $this->dev5->language->get('field_product_d5bling_tributacao_spedTipoItem_opt_00'),
                                            '01' => $this->dev5->language->get('field_product_d5bling_tributacao_spedTipoItem_opt_01'),
                                            '02' => $this->dev5->language->get('field_product_d5bling_tributacao_spedTipoItem_opt_02'),
                                            '03' => $this->dev5->language->get('field_product_d5bling_tributacao_spedTipoItem_opt_03'),
                                            '04' => $this->dev5->language->get('field_product_d5bling_tributacao_spedTipoItem_opt_04'),
                                            '05' => $this->dev5->language->get('field_product_d5bling_tributacao_spedTipoItem_opt_05'),
                                            '06' => $this->dev5->language->get('field_product_d5bling_tributacao_spedTipoItem_opt_06'),
                                            '07' => $this->dev5->language->get('field_product_d5bling_tributacao_spedTipoItem_opt_07'),
                                            '08' => $this->dev5->language->get('field_product_d5bling_tributacao_spedTipoItem_opt_08'),
                                            '09' => $this->dev5->language->get('field_product_d5bling_tributacao_spedTipoItem_opt_09'),
                                            '10' => $this->dev5->language->get('field_product_d5bling_tributacao_spedTipoItem_opt_10'),
                                            '99' => $this->dev5->language->get('field_product_d5bling_tributacao_spedTipoItem_opt_99')
                                        ]
                                    ], // d5bling[tributacao][spedTipoItem]
                                    'codigoItem' => 'text', // d5bling[tributacao][codigoItem]
                                    'percentualTributos' => 'number', // d5bling[tributacao][percentualTributos]
                                    'valorBaseStRetencao' => 'number', // d5bling[tributacao][valorBaseStRetencao]
                                    'valorStRetencao' => 'number', // d5bling[tributacao][valorStRetencao]
                                    'valorICMSSubstituto' => 'number', // d5bling[tributacao][valorICMSSubstituto]
                                    'codigoExcecaoTipi' => 'text', // d5bling[tributacao][codigoExcecaoTipi]
                                    'classeEnquadramentoIpi' => 'text', // d5bling[tributacao][classeEnquadramentoIpi]
                                    'valorIpiFixo' => 'number', // d5bling[tributacao][valorIpiFixo]
                                    'codigoSeloIpi' => [
                                        'type' => 'select', // d5bling[tributacao][codigoSeloIpi]
                                        'options' => [
                                            '' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_none'),
                                            '8610-09' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_8610-09'),
                                            '9710-01' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9710-01'),
                                            '9710-10' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9710-10'),
                                            '9710-11' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9710-11'),
                                            '9710-12' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9710-12'),
                                            '9729-11' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9729-11'),
                                            '9729-12' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9729-12'),
                                            '9729-13' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9729-13'),
                                            '9729-14' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9729-14'),
                                            '9729-21' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9729-21'),
                                            '9729-22' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9729-22'),
                                            '9729-23' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9729-23'),
                                            '9729-24' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9729-24'),
                                            '9737-11' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9737-11'),
                                            '9737-12' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9737-12'),
                                            '9737-13' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9737-13'),
                                            '9737-14' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9737-14'),
                                            '9737-15' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9737-15'),
                                            '9737-16' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9737-16'),
                                            '9737-21' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9737-21'),
                                            '9737-22' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9737-22'),
                                            '9737-23' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9737-23'),
                                            '9745-11' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9745-11'),
                                            '9745-12' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9745-12'),
                                            '9745-13' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9745-13'),
                                            '9822-11' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9822-11'),
                                            '9822-12' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9822-12'),
                                            '9822-13' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9822-13'),
                                            '9829-11' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9829-11'),
                                            '9829-12' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9829-12'),
                                            '9829-13' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9829-13'),
                                            '9829-14' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9829-14'),
                                            '9829-21' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9829-21'),
                                            '9829-22' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9829-22'),
                                            '9829-23' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9829-23'),
                                            '9829-24' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9829-24'),
                                            '9837-11' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9837-11'),
                                            '9837-12' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9837-12'),
                                            '9837-13' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9837-13'),
                                            '9837-14' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9837-14'),
                                            '9837-15' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9837-15'),
                                            '9837-16' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9837-16'),
                                            '9837-21' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9837-21'),
                                            '9837-22' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9837-22'),
                                            '9837-23' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9837-23'),
                                            '9845-11' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9845-11'),
                                            '9845-12' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9845-12'),
                                            '9845-13' => $this->dev5->language->get('field_product_d5bling_tributacao_codigoSeloIpi_opt_9845-13'),
                                        ]
                                    ],
                                    'valorPisFixo' => 'number', // d5bling[tributacao][valorPisFixo]
                                    'valorCofinsFixo' => 'number', // d5bling[tributacao][valorCofinsFixo]
                                    'codigoANP' => 'text', // d5bling[tributacao][codigoANP]
                                    'descricaoANP' => 'text', // d5bling[tributacao][descricaoANP]
                                    'percentualGLP' => 'number', // d5bling[tributacao][percentualGLP]
                                    'percentualGasNacional' => 'number', // d5bling[tributacao][percentualGasNacional]
                                    'percentualGasImportado' => 'number', // d5bling[tributacao][percentualGasImportado]
                                    'valorPartida' => 'number', // d5bling[tributacao][valorPartida]
                                    'tipoArmamento' => [ // d5bling[tributacao][tipoArmamento]
                                        'type' => 'select',
                                        'options' => [
                                            '' => $this->dev5->language->get('field_product_d5bling_tributacao_tipoArmamento_opt_none'),
                                            0 => $this->dev5->language->get('field_product_d5bling_tributacao_tipoArmamento_opt_0'),
                                            1 => $this->dev5->language->get('field_product_d5bling_tributacao_tipoArmamento_opt_1')
                                        ]
                                    ],
                                    'descricaoCompletaArmamento' => 'text', // d5bling[tributacao][descricaoCompletaArmamento]
                                    'dadosAdicionais' => 'text', // d5bling[tributacao][dadosAdicionais]
                                    'grupoProduto' => [
                                        'type' => 'level',
                                        'level' => true,
                                        'fields' => [
                                            'id' => [ // d5bling[tributacao][grupoProduto][id]
                                                'type' => 'autocomplete',
                                                'callback' => function ($value) {
                                                    $value = explode(' → ', $value);
                                                    $filter = count($value) > 1
                                                        ? ['nomePai' => trim($value[0]) ?: null, 'nome' => trim($value[1]) ?: null]
                                                        : ['nome' => trim($value[0]) ?: null];

                                                    return ['' => ''] + array_column(
                                                            array_map(static function ($i) {
                                                                $i['full_name'] = !empty($i['grupoProdutoPai']['nome'])
                                                                    ? "{$i['grupoProdutoPai']['nome']} → $i[nome]"
                                                                    : $i['nome'];

                                                                $i['id'] = "$i[id]";
                                                                return $i;
                                                            }, Api::getInstance()->getGruposProdutos($filter, ['cache' => 7 * 24 * 60 * 60])),
                                                            'full_name',
                                                            'id'
                                                        );
                                                },
                                            ],
                                        ]
                                    ]
                                ]
                            ],
                            'midia' => [
                                'type' => 'level',
                                'level' => true,
                                'fields' => [
                                    'video' => [
                                        'type' => 'level',
                                        'level' => true,
                                        'fields' => [
                                            'url' => 'text', // d5bling[midia][video][url]
                                        ]
                                    ],
                                    // midia[imagens][imagensURL][*][link], string
                                ]
                            ],
                            // linhaProduto, number
                            // estrutura[tipoEstoque], enum[F, V]
                            // estrutura[lancamentoEstoque], enum[P, M, A]
                            // estrutura[componentes][*][id], int
                            // estrutura[componentes][*][quantidade], int
                            'campos_customizados' => function () {
                                $campos_customizados = array_column(array_filter(array_map(function ($i) {
                                    if ($i['situacao'] !== 1) {
                                        return false;
                                    }

                                    $this->dev5->language->set("field_product_d5bling_campos_customizados_$i[id]", $i['nome']);
                                    $this->dev5->language->set("field_product_d5bling_campos_customizados_$i[id]_invalid", $this->dev5->language->get('field_invalid_required'));
                                    switch ($i['tipoCampo']['id'] = intval(@$i['tipoCampo']['id'])) {
                                        case 3: // Lista
                                            return [
                                                'campos_customizados_id' => $i['id'],
                                                'type' => 'select',
                                                'options' => array_column($i['opcoes'], 'nome', 'id')
                                            ];
                                        case 1: // Inteiro
                                        case 2: // Decimal
                                        case 4: // Texto
                                        case 5: // Texto Longo
                                        case 6: // Lógico
                                        case 7: // Data
                                            $i['tipoCampo']['id'] === 1 && $type = 'number';
                                            $i['tipoCampo']['id'] === 2 && $type = 'number';
                                            $i['tipoCampo']['id'] === 4 && $type = 'text';
                                            $i['tipoCampo']['id'] === 5 && $type = 'textarea';
                                            $i['tipoCampo']['id'] === 6 && $type = 'yesno';
                                            $i['tipoCampo']['id'] === 7 && $type = 'date';

                                            $attrs = [];
                                            $attrs['placeholder'] = $i['placeholder'] ?: $i['nome'];
                                            if (!empty($i['tamanho']['minimo'])) {
                                                if ($i['tipoCampo']['id'] === 4 && $i['tipoCampo']['id'] === 5) {
                                                    $attrs['minlength'] = $i['tamanho']['minimo'];
                                                } else {
                                                    $attrs['min'] = $i['tamanho']['minimo'];
                                                }
                                            }
                                            if (!empty($i['tamanho']['maximo'])) {
                                                if ($i['tipoCampo']['id'] === 4 && $i['tipoCampo']['id'] === 5) {
                                                    $attrs['maxlength'] = $i['tamanho']['maximo'];
                                                } else {
                                                    $attrs['max'] = $i['tamanho']['maximo'];
                                                }
                                            }

                                            return [
                                                'campos_customizados_id' => $i['id'],
                                                'type' => $type,
                                                'attrs' => $attrs,
                                                'invalidate' => !!$i['obrigatorio']
                                            ];
                                    }
                                    return false;
                                }, Api::getInstance()->getCamposCustomizadosModulo('Produtos', ['cache' => 7 * 24 * 60 * 60]))), null, 'campos_customizados_id');

                                return !$campos_customizados ? false : [
                                    'type' => 'section',
                                    'level' => true,
                                    'fields' => $campos_customizados
                                ];
                            }
                        ],
                        'value' => $values
                    ]
                    // phpcs:enable
                ], ['admin' => false])
                . '</div>'
                . $this->dev5->view([
                    'flag' => $this->dev5->asset('bling/logo.webp'),
                    'language_id' => $this->dev5->field('language_id'),
                    'code_mode' => $this->dev5->field('code_mode'),
                    'view_mode' => $this->dev5->field('view_mode'),
                    'error' => $error
                ], $this->dev5->route() . "_product_form")
        ];
    }

    /**
     * @param array $error
     * @return void
     */
    public function productValidate(&$error = [])
    {
        if (!$this->dev5->status()) {
            return;
        }

        $input = &$this->request->post['product_d5bling'];
        if (empty($input['deposito'])) {
            $input['deposito'] = $this->dev5->field('deposito'); // Set default deposito
        }

        if ($this->dev5->field('code_mode') === 'sku') {
            if (
                !$this->productValidateSKU([
                    'product_id' => isset($this->request->get['product_id']) ? $this->request->get['product_id'] : 0,
                    'sku' => @$this->request->post['sku']
                ])
            ) {
                $error['sku'] = $this->dev5->language->get('field_invalid');
            }

            if (($column = ProductEntity::getInstance()->getColumnOptionSKU()) !== 'NULL') {
                if (!empty($this->request->post['product_option'])) {
                    foreach ($this->request->post['product_option'] as $option_row_id => $option) {
                        if (empty($option['product_option_value'])) {
                            continue;
                        }

                        foreach ($option['product_option_value'] as $option_value_row_id => $option_value) {
                            if (
                                !$this->productValidateSKU([
                                    'product_id' => isset($this->request->get['product_id']) ? $this->request->get['product_id'] : 0, // phpcs:ignore
                                    'product_option_value_id' => isset($option_value['product_option_value_id']) ? $option_value['product_option_value_id'] : 0, // phpcs:ignore
                                    'column' => $column,
                                    'sku' => @$option_value['sku']
                                ])
                            ) {
                                $error["sku_option-$option_row_id-$option_value_row_id"] = $this->dev5->language->get('field_invalid'); // phpcs:ignore
                            }
                        }
                    }
                }
            }
        }

        foreach (Api::getInstance()->getCamposCustomizadosModulo('Produtos', ['cache' => 7 * 24 * 60 * 60]) as $i) {
            if (empty($i['obrigatorio']) || !empty($input['campos_customizados'][$i['id']])) {
                continue;
            }

            $error['product_d5bling']['campos_customizados'][$i['id']] = $this->dev5->language->get('field_invalid');
        }

        if (!empty($error) && empty($error['warning'])) {
            $error['warning'] = $this->dev5->language->get('error_warning');
        }
    }

    /**
     * @param array $data
     * @return bool
     */
    private function productValidateSKU($data)
    {
        if (empty($data['sku'])) {
            return false;
        }

        return !$this->dev5->db->query("
            SELECT
                COUNT(*) as total
            FROM
                " . DB_PREFIX . "product p
            WHERE
                p.sku = :sku
                AND p.product_id != :product_id
        ", [
            'product_id' => array_key_exists('product_option_value_id', $data) ? -1 : $data['product_id'],
            'sku' => $data['sku']
        ])->column() && !$this->dev5->db->query("
            SELECT
                COUNT(*) as total
            FROM
                " . DB_PREFIX . "product_option_value pov
            WHERE
                {$this->dev5->db->escape($data['column'], true)} = :sku
                AND product_option_value_id != :product_option_value_id
        ", [
            'product_option_value_id' => array_key_exists('product_option_value_id', $data) ? $data['product_option_value_id'] : -1, // phpcs:ignore
            'sku' => $data['sku']
        ])->column();
    }

    /**
     * @param string $event
     * @param int $product_id
     * @return void
     */
    public function productEvent($event, $product_id = null)
    {
        if (ProductEntity::ignore()) {
            return;
        }

        if (!$this->dev5->status()) {
            return;
        }

        if (empty($product_id) || !is_numeric($product_id)) {
            return;
        }

        if (!in_array($event, @$this->dev5->field('ei')['export_product'] ?: [])) {
            return;
        }

        if ($event !== 'delete') {
            if (!isset($this->request->post['product_d5bling'])) {
                return;
            }

            empty($product_id) || $entity = ProductEntity::findOne(['product_id' => $product_id, 'options' => '']);
            empty($entity) && $entity = new ProductEntity();

            $save = $this->request->post['product_d5bling'];
            $save['product_id'] = $product_id;
            $save['options'] = '';
            $save['bling_id'] = $entity->bling_id;
            ProductEntity::fromArray($save)->save(['returning' => false]);
        }

        QueueEntity::publish("export_product:$product_id", ['event' => $event]);
    }

    /**
     * @param string $event
     * @param array $filter
     * @param array $arg
     * @return mixed
     */
    public function customerEvent($event, $filter = null, &$arg = null)
    {
        if (!$this->dev5->status()) {
            return false;
        }

        if (in_array($this->request->get['route'], ['sale/custom_field', 'customer/custom_field', 'customer/custom_field.list'])) { // phpcs:ignore
            return false; // Not list as a real customer custom field
        }

        switch ($event) {
            case 'fields':
                if (!isset($filter['filter_location']) || $filter['filter_location'] === 'account') {
                    $arg[] = [
                        'custom_field_id' => 'customer_d5bling_seller',
                        'type' => 'select',
                        'value' => '',
                        'validation' => '',
                        'location' => 'account',
                        'status' => 1,
                        'sort_order' => 999,
                        'language_id' => $this->config->get('config_language_id'),
                        'name' => in_array($this->request->get['route'], ['sale/order/info', 'sale/order.info'])
                            ? $this->dev5->language->get('field_customer_seller_inner')
                            : $this->dev5->language->get('field_customer_seller'),
                        'required' => false
                    ];
                    return true;
                }
                break;

            case 'field_values':
                if ($filter === 'customer_d5bling_seller') {
                    return array_map(static function ($seller) {
                        return [
                            'custom_field_value_id' => "d5bling_seller_$seller[id]",
                            'name' => $seller['contato']['nome']
                        ];
                    }, Api::getInstance()->getVendedores(['situacaoContato' => 'A'], ['cache' => 7 * 24 * 60 * 60]));
                }
                break;

            case 'field_value':
                if (str_starts_with($filter, 'd5bling_seller_')) {
                    return array_find(
                        array_map(static function ($seller) {
                            return [
                                'custom_field_value_id' => "d5bling_seller_$seller[id]",
                                'name' => $seller['contato']['nome']
                            ];
                        }, Api::getInstance()->getVendedores(['situacaoContato' => 'A'], ['cache' => 7 * 24 * 60 * 60])), // phpcs:ignore
                        static function ($seller) use ($filter) {
                            return $seller['custom_field_value_id'] === $filter;
                        }
                    );
                }
                break;
        }

        return false;
    }

    /**
     * @param string $default
     * @param bool $invalidate
     * @return array
     */
    private function getCustomerFields($default, $invalidate = true)
    {
        static $customer_fields = null;
        if ($customer_fields === null) {
            // phpcs:disable
            $this->load->language(ocver('2.1', '<') ? 'sale/customer' : 'customer/customer');

            $customer_fields = [];
            $customer_fields[$this->dev5->language->get('text_customer_opt_special')] = [
                'customer_id' => $this->dev5->language->get('text_customer_opt_customer_id'),
                'customer' => $this->dev5->language->get('text_customer_opt_customer'),

                'payment_customer' => "{$this->dev5->language->get('text_customer_opt_address_payment')} → {$this->dev5->language->get('text_customer_opt_customer')}",
                'payment_zone_code' => "{$this->dev5->language->get('text_customer_opt_address_payment')} → {$this->dev5->language->get('text_customer_opt_zone_code')}",

                'shipping_customer' => "{$this->dev5->language->get('text_customer_opt_address_shipping')} → {$this->dev5->language->get('text_customer_opt_customer')}",
                'shipping_zone_code' => "{$this->dev5->language->get('text_customer_opt_address_shipping')} → {$this->dev5->language->get('text_customer_opt_zone_code')}"
            ];
            $customer_fields[$this->dev5->language->get('text_customer_opt_default')] = [
                'firstname' => $this->language->get('entry_firstname'),
                'lastname' => $this->language->get('entry_lastname'),
                'email' => $this->language->get('entry_email'),
                'telephone' => $this->language->get('entry_telephone'),

                'payment_firstname' => "{$this->dev5->language->get('text_customer_opt_address_payment')} → {$this->language->get('entry_firstname')}",
                'payment_lastname' => "{$this->dev5->language->get('text_customer_opt_address_payment')} → {$this->language->get('entry_lastname')}",
                'payment_company' => "{$this->dev5->language->get('text_customer_opt_address_payment')} → {$this->language->get('entry_company')}",
                'payment_address_1' => "{$this->dev5->language->get('text_customer_opt_address_payment')} → {$this->language->get('entry_address_1')}",
                'payment_address_2' => "{$this->dev5->language->get('text_customer_opt_address_payment')} → {$this->language->get('entry_address_2')}",
                'payment_city' => "{$this->dev5->language->get('text_customer_opt_address_payment')} → {$this->language->get('entry_city')}",
                'payment_postcode' => "{$this->dev5->language->get('text_customer_opt_address_payment')} → {$this->language->get('entry_postcode')}",
                'payment_country' => "{$this->dev5->language->get('text_customer_opt_address_payment')} → {$this->language->get('entry_country')}",
                'payment_zone' => "{$this->dev5->language->get('text_customer_opt_address_payment')} → {$this->language->get('entry_zone')}",

                'shipping_firstname' => "{$this->dev5->language->get('text_customer_opt_address_shipping')} → {$this->language->get('entry_firstname')}",
                'shipping_lastname' => "{$this->dev5->language->get('text_customer_opt_address_shipping')} → {$this->language->get('entry_lastname')}",
                'shipping_company' => "{$this->dev5->language->get('text_customer_opt_address_shipping')} → {$this->language->get('entry_company')}",
                'shipping_address_1' => "{$this->dev5->language->get('text_customer_opt_address_shipping')} → {$this->language->get('entry_address_1')}",
                'shipping_address_2' => "{$this->dev5->language->get('text_customer_opt_address_shipping')} → {$this->language->get('entry_address_2')}",
                'shipping_city' => "{$this->dev5->language->get('text_customer_opt_address_shipping')} → {$this->language->get('entry_city')}",
                'shipping_postcode' => "{$this->dev5->language->get('text_customer_opt_address_shipping')} → {$this->language->get('entry_postcode')}",
                'shipping_country' => "{$this->dev5->language->get('text_customer_opt_address_shipping')} → {$this->language->get('entry_country')}",
                'shipping_zone' => "{$this->dev5->language->get('text_customer_opt_address_shipping')} → {$this->language->get('entry_zone')}",
            ];
            $customer_fields[$this->dev5->language->get('text_customer_opt_custom')] = [];
            $this->load->model(ocVer('2.1', '<') ? 'sale/custom_field' : 'customer/custom_field');
            $custom_fields = $this->{'model_' . (ocVer('2.1', '<') ? 'sale_custom_field' : 'customer_custom_field')}->getCustomFields(['sort' => 'cfd.name']);

            foreach ($custom_fields as $custom_field) {
                if ($custom_field['status'] && $custom_field['location'] != 'address') {
                    $customer_fields[$this->dev5->language->get('text_customer_opt_custom')]["custom_field_$custom_field[custom_field_id]"] = $custom_field['name'];
                }
            }
            foreach (['payment', 'shipping'] as $k) {
                foreach ($custom_fields as $custom_field) {
                    if ($custom_field['status'] && $custom_field['location'] == 'address') {
                        $customer_fields[$this->dev5->language->get('text_customer_opt_custom')]["{$k}_custom_field_$custom_field[custom_field_id]"] = "{$this->dev5->language->get("text_customer_opt_address_$k")} → $custom_field[name]";
                    }
                }
            }
            // phpcs:enable
        }

        $options = !$invalidate
            ? array_merge(['' => $this->dev5->language->get('text_customer_opt_none')], $customer_fields)
            : $customer_fields;

        $option_default = "";
        if ($default) {
            foreach ($options as $value => $label) {
                is_string($label) && $label = [$value => $label];
                foreach ($label as $sub_value => $sub_label) {
                    if (!preg_match($default, $sub_label)) {
                        continue;
                    }

                    $option_default = $sub_value;
                    break 2;
                }
            }
        }

        return [
            'type' => 'select',
            'options' => $options,
            'default' => $option_default,
            'invalidate' => $invalidate
        ];
    }

    /**
     * @param string $value
     * @return array<string, string>
     */
    private function autocompletePaymentMethods($value)
    {
        static $json = null;

        if (null === $json) {
            $json = [];

            $query = $this->dev5->db->querySelect(DB_PREFIX . 'extension', [
                'distinct' => true,
                'where' => ['type' => 'payment']
            ]);

            foreach ($query as $i) {
                if (isset($json[$i['code']])) {
                    continue;
                }

                $language = ocver('2.3', '<')
                    ? "payment/$i[code]"
                    : (ocver('4', '<')
                        ? "extension/payment/$i[code]"
                        : "extension/$i[extension]/payment/$i[code]"
                    );

                $this->dev5->language->set('heading_title', null);
                $json[$i['code']] = strip_tags(@$this->load->language($language)['heading_title'] ?: $i['code']);
                $json[$i['code']] !== $i['code'] && $json[$i['code']] .= " ($i[code])";
            }

            $query = $this->dev5->db->querySelect(DB_PREFIX . 'order', [
                'distinct' => true,
                'columns' => ocver('4.0.2', '>=') ? ['payment_method'] : ['payment_code', 'payment_method'],
                'group_by' => ocver('4.0.2', '>=') ? 'payment_method' : 'payment_code',
                'order_by' => 'order_id'
            ]);

            foreach ($query as $i) {
                if (ocver('4.0.2', '>=')) {
                    $i = json_decode($i['payment_method'], true);
                    $i['payment_code'] = $i['code'];
                    $i['payment_method'] = $i['name'];
                }

                $json[$i['payment_code']] = @$i['payment_method'] ?: $i['payment_code'];
                $json[$i['payment_code']] !== $i['payment_code'] && $json[$i['payment_code']] .= " ($i[payment_code])";
            }
        }

        $value = strtolower("$value");
        return ['' => ''] + array_filter($json, static function ($label) use ($value) {
                return !$value || str_contains(strtolower($label), $value);
        });
    }

    /**
     * @param string $value
     * @return array<string, string>
     */
    private function autocompleteShippingMethods($value)
    {
        static $json = null;

        if (null === $json) {
            $json = [];

            $query = $this->dev5->db->querySelect(DB_PREFIX . 'extension', [
                'distinct' => true,
                'where' => ['type' => 'shipping']
            ]);

            foreach ($query as $i) {
                $code = "$i[code].*";

                if (isset($json[$code])) {
                    continue;
                }

                $language = ocver('2.3', '<')
                    ? "shipping/$i[code]"
                    : (ocver('4', '<')
                        ? "extension/shipping/$i[code]"
                        : "extension/$i[extension]/shipping/$i[code]"
                    );

                $this->dev5->language->set('heading_title', null);
                $json[$code] = strip_tags(@$this->load->language($language)['heading_title'] ?: $code);
                $json[$code] !== $code && $json[$code] .= " ($code)";
            }

            $query = $this->dev5->db->querySelect(DB_PREFIX . 'order', [
                'distinct' => true,
                'columns' => ocver('4.0.2', '>=') ? ['shipping_method'] : ['shipping_code', 'shipping_method'],
                'group_by' => ocver('4.0.2', '>=') ? 'shipping_method' : 'shipping_code',
                'order_by' => 'order_id'
            ]);

            foreach ($query as $i) {
                if (ocver('4.0.2', '>=')) {
                    $i = json_decode($i['shipping_method'], true);
                    $i['shipping_code'] = $i['code'];
                    $i['shipping_method'] = $i['name'];
                }

                if (stripos($i['shipping_code'], '.') === false) {
                    $i['shipping_code'] .= '.*';
                }

                if (isset($shipping_methods[$i['shipping_code']])) {
                    continue;
                }

                $json[$i['shipping_code']] = @$i['shipping_method'] ?: $i['shipping_code'];
                $json[$i['shipping_code']] !== $i['shipping_code'] && $json[$i['shipping_code']] .= " ($i[shipping_code])"; // phpcs:ignore
            }
        };

        $value = strtolower($value);
        return ['' => ''] + array_filter($json, static function ($label) use ($value) {
                return !$value || str_contains(strtolower($label), $value);
        });
    }

    /**
     * @param string $value
     * @return array<string, string>
     */
    private function autocompleteCategories($value)
    {
        return ['' => ''] + array_column($this->model_catalog_category->getCategories([
                'filter_name' => trim(preg_replace('/\s+/', '%', $value)),
                'sort' => 'name'
            ]), 'name', 'category_id');
    }

    /**
     * @param string $value
     * @return array<string, string>
     */
    private function autocompleteProducts($value)
    {
        return ['' => ''] + array_column($this->model_catalog_product->getProducts([
                'filter_name' => trim(preg_replace('/\s+/', '%', $value)),
                'sort' => 'name'
            ]), 'name', 'product_id');
    }

    /**
     * @return void
     */
    public function eiEvent()
    {
        switch ($type = @$this->request->get['type']) {
            case 'export_categories':
            case 'export_products':
            case 'export_orders':
            case 'import_categories':
            case 'import_products':
            case 'import_orders':
                break;
            default:
                http_response_code(400);
                exit;
        }

        switch ($event = @$this->request->get['event']) {
            case 'cancel':
                $this->eiCancel($type);
                return;

            case 'getting-form':
                $this->eiGettingForm($type);
                return;

            case 'getting-submit':
                $this->eiGettingSubmit($type);
                return;

            case 'approve-form':
                $this->eiApproveForm($type);
                return;

            case 'approve-submit':
                $this->eiApproveSubmit($type);
                return;

            case 'getting-next-form':
            case 'approve-next-form':
                $this->eiNextForm($type, $event);
                return;

            case 'getting-next':
            case 'approve-next':
                $payload = EIEntity\Payload::currentPayload();

                if ($payload->type !== $type) {
                    $payload->status = EIEntity\Payload::NONE;
                } elseif ($payload->status === EIEntity\Payload::GETTING_WAITING) {
                    $this->eiGettingNext($payload);
                } elseif ($payload->status === EIEntity\Payload::APPROVE_WAITING) {
                    $this->eiApproveNext($payload);
                }

                header('Content-Type: application/json; charset=utf-8');
                echo json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
                exit;

            case 'summary':
                $this->eiSummary($type);
                return;

            default:
                http_response_code(400);
                exit;
        }
    }

    /**
     * @param string $type
     * @return void
     */
    private function eiCancel($type)
    {
        if (EIEntity\Payload::currentPayload()->type === $type) {
            $empty = new EIEntity\Payload();
            $empty->save();
            EIEntity::truncate();
        }

        RWin::error('dev5-bling-ei', ['message' => 'canceled by user']);
        exit;
    }

    /**
     * @param string $type
     * @return void
     */
    private function eiGettingForm($type)
    {
        switch ($type) {
            // phpcs:disable
            case 'export_categories':
                $fields = [
                    'filter' => [
                        'type' => 'section',
                        'fields' => [
                            'id' => 'well-number',
                            'name' => [
                                'type' => 'well-autocomplete',
                                'callback' => function ($value) {
                                    return $this->autocompleteCategories($value);
                                }
                            ],
                            'status' => [
                                'type' => 'select',
                                'options' => [
                                    '' => $this->dev5->language->get('field_ei_export_categories_status_opt_none'),
                                    '1' => $this->dev5->language->get('text_enabled'),
                                    '0' => $this->dev5->language->get('text_disabled')
                                ]
                            ],
                            'date_added' => [
                                'type' => 'input-group',
                                'level' => true,
                                'fields' => [
                                    'start' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_start')]
                                    ],
                                    '-',
                                    'end' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_end')]
                                    ]
                                ]
                            ],
                            'date_modified' => [
                                'type' => 'input-group',
                                'level' => true,
                                'fields' => [
                                    'start' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_start')]
                                    ],
                                    '-',
                                    'end' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_end')]
                                    ]
                                ]
                            ]
                        ]
                    ]
                ];
                break;
            case 'export_products':
                $fields = [
                    'filter' => [
                        'type' => 'section',
                        'fields' => [
                            'id' => 'well-number',
                            'name' => [
                                'type' => 'well-autocomplete',
                                'callback' => function ($value) {
                                    return $this->autocompleteProducts($value);
                                }
                            ],
                            'status' => [
                                'type' => 'select',
                                'options' => [
                                    '' => $this->dev5->language->get('field_ei_export_products_status_opt_none'),
                                    '1' => $this->dev5->language->get('text_enabled'),
                                    '0' => $this->dev5->language->get('text_disabled')
                                ]
                            ],
                            'date_added' => [
                                'type' => 'input-group',
                                'level' => true,
                                'fields' => [
                                    'start' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_start')]
                                    ],
                                    '-',
                                    'end' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_end')]
                                    ]
                                ]
                            ],
                            'date_modified' => [
                                'type' => 'input-group',
                                'level' => true,
                                'fields' => [
                                    'start' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_start')]
                                    ],
                                    '-',
                                    'end' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_end')]
                                    ]
                                ]
                            ]
                        ]
                    ]
                ];
                break;
            case 'export_orders':
                $fields = [
                    'filter' => [
                        'type' => 'section',
                        'fields' => [
                            'id' => 'well-number',
                            'status' => [
                                'type' => 'well-checkbox',
                                'options' => array_column(
                                    $this->model_localisation_order_status->getOrderStatuses(),
                                    'name',
                                    'order_status_id'
                                )
                            ],
                            'date_added' => [
                                'type' => 'input-group',
                                'level' => true,
                                'fields' => [
                                    'start' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_start')]
                                    ],
                                    '-',
                                    'end' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_end')]
                                    ]
                                ]
                            ],
                            'date_modified' => [
                                'type' => 'input-group',
                                'level' => true,
                                'fields' => [
                                    'start' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_start')]
                                    ],
                                    '-',
                                    'end' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_end')]
                                    ]
                                ]
                            ]
                        ]
                    ]
                ];
                break;
            case 'import_categories':
                $fields = [
                    'filter' => [
                        'type' => 'section',
                        'fields' => [
                            'bling_id' => 'well-number',
                            'name' => [
                                'type' => 'well-autocomplete',
                                'callback' => function ($value) {
                                    return Api::getInstance()->getCategoriasProdutosNormalized($value, ['cache' => 7 * 24 * 60 * 60]) ?: [];
                                }
                            ]
                        ]
                    ]
                ];
                break;
            case 'import_products':
                $fields = [
                    'filter' => [
                        'type' => 'section',
                        'fields' => [
                            'bling_id' => 'well-number',
                            'name' => [
                                'type' => 'well-autocomplete',
                                'callback' => function ($value) {
                                    return Api::getInstance()->getProdutosNormalized($value, ['cache' => 7 * 24 * 60 * 60]) ?: [];
                                }
                            ],
                            'codigo' => 'well-text',
                            'status' => [
                                'type' => 'select',
                                'options' => [
                                    '' => $this->dev5->language->get('field_ei_import_products_status_opt_none'),
                                    '2' => $this->dev5->language->get('text_enabled'),
                                    '3' => $this->dev5->language->get('text_disabled')
                                ]
                            ],
                            'date_added' => [
                                'type' => 'input-group',
                                'level' => true,
                                'fields' => [
                                    'start' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_start')]
                                    ],
                                    '-',
                                    'end' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_end')]
                                    ]
                                ]
                            ],
                            'date_modified' => [
                                'type' => 'input-group',
                                'level' => true,
                                'fields' => [
                                    'start' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_start')]
                                    ],
                                    '-',
                                    'end' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_end')]
                                    ]
                                ]
                            ],
                        ]
                    ],
                    'config' => [
                        'type' => 'section',
                        'fields' => [
                            'found' => [
                                'type' => 'select',
                                'options' => [
                                    'ignore' => $this->dev5->language->get('field_ei_import_products_found_opt_ignore'),
                                    'stock' => $this->dev5->language->get('field_ei_import_products_found_opt_stock'),
                                    'replace' => $this->dev5->language->get('field_ei_import_products_found_opt_replace')
                                ],
                                'value' => 'stock'
                            ],
                            'not_found' => [
                                'type' => 'select',
                                'options' => [
                                    '' => $this->dev5->language->get('field_ei_import_products_not_found_opt_ignore'),
                                    'create' => $this->dev5->language->get('field_ei_import_products_not_found_opt_create')
                                ],
                                'value' => 'create'
                            ],
                            'category' => [
                                'type' => 'select',
                                'options' => [
                                    'ignore' => $this->dev5->language->get('field_ei_import_products_category_opt_ignore'),
                                    'create' => $this->dev5->language->get('field_ei_import_products_category_opt_create')
                                ],
                                'value' => 'create'
                            ],
                            'image' => [
                                'type' => 'select-yesno',
                                'value' => '1'
                            ]
                        ]
                    ]
                ];
                break;
            case 'import_orders':
                $fields = [
                    'filter' => [
                        'type' => 'section',
                        'fields' => [
                            'bling_id' => 'well-number',
                            'status' => [
                                'type' => 'well-checkbox',
                                'options' => $options = array_column(
                                    Api::getInstance()->getSituacoesModulo('Vendas', ['cache' => 7 * 24 * 60 * 60]),
                                    'nome',
                                    'id'
                                ),
                                'value' => array_keys($options)
                            ],
                            'numero_loja' => 'well-number',
                            'date_added' => [
                                'type' => 'input-group',
                                'level' => true,
                                'fields' => [
                                    'start' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_start')]
                                    ],
                                    '-',
                                    'end' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_end')]
                                    ]
                                ]
                            ],
                            'date_modified' => [
                                'type' => 'input-group',
                                'level' => true,
                                'fields' => [
                                    'start' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_start')]
                                    ],
                                    '-',
                                    'end' => [
                                        'type' => 'date',
                                        'attrs' => ['placeholder' => $this->dev5->language->get('text_date_end')]
                                    ]
                                ]
                            ],
                            'channel_id' => [
                                'type' => 'select',
                                'options' =>
                                    ['' => $this->dev5->language->get('field_ei_import_orders_channel_id_opt_none')]
                                    + array_column(
                                        Api::getInstance()->getCanaisVenda(['cache' => 7 * 24 * 60 * 60]),
                                        'descricao',
                                        'id'
                                    ),
                                'value' => ''
                            ]
                        ]
                    ],
                    'config' => [
                        'type' => 'section',
                        'fields' => [
                            'found' => [
                                'type' => 'select',
                                'options' => [
                                    'ignore' => $this->dev5->language->get('field_ei_import_orders_found_opt_ignore'),
                                    'status' => $this->dev5->language->get('field_ei_import_orders_found_opt_status'),
                                    'replace' => $this->dev5->language->get('field_ei_import_orders_found_opt_replace')
                                ],
                                'value' => 'status'
                            ],
                            'not_found' => [
                                'type' => 'select',
                                'options' => [
                                    'ignore' => $this->dev5->language->get('field_ei_import_orders_not_found_opt_ignore'),
                                    'create' => $this->dev5->language->get('field_ei_import_orders_not_found_opt_create')
                                ],
                                'value' => 'create'
                            ],
                        ]
                    ]
                ];
                break;
            // phpcs:enable

            default:
                http_response_code(400);
                exit;
        }

        $continue = false;
        $payload = EIEntity\Payload::currentPayload();
        if ($payload->type === $type && $payload->status !== EIEntity\Payload::NONE) {
            switch ($payload->status) {
                case EIEntity\Payload::GETTING_WAITING:
                case EIEntity\Payload::GETTING_RUNNING:
                    $continue = 'getting-next-form';
                    break;

                case EIEntity\Payload::GETTING_FINISHED:
                    $continue = 'approve-form';
                    break;

                case EIEntity\Payload::APPROVE_WAITING:
                case EIEntity\Payload::APPROVE_RUNNING:
                    $continue = 'approve-next-form';
                    break;
            }
        }

        $this->document->setTitle($this->dev5->language->get("heading_title_ei_$type"));
        $this->response->setOutput($this->dev5->view([
            'event' => 'form',
            'type' => $type,
            'form' => $this->dev5->form([
                $type => ['type' => 'level', "level" => ["ei", $type], 'fields' => $fields]
            ], ['admin' => false]),
            'href_ei_event' => $this->dev5->link('eiEvent', ['type' => $type]),
            'continue' => $continue,
            'header' => $this->load->controller('common/header'),
            'footer' => $this->load->controller('common/footer')
        ], $this->dev5->route() . "_ei_form"));
    }

    /**
     * @param string $type
     * @return void
     */
    private function eiGettingSubmit($type)
    {
        $payload = new EIEntity\Payload();
        $payload->status = EIEntity\Payload::GETTING_WAITING;
        $payload->type = $type;
        $payload->input = @$this->request->post['ei'][$type];
        $payload->save();

        EIEntity::truncate();

        $this->response->redirect($this->dev5->link('eiEvent', ['type' => $type, 'event' => 'getting-next-form'])); // phpcs:ignore
        exit;
    }

    /**
     * @param EIEntity\Payload $payload
     * @return void
     */
    private function eiGettingNext($payload)
    {
        $this->dev5->debug("• EI-GETTING-NEXT", "type=$payload->type", "cursor=$payload->cursor");

        if ($payload->status !== EIEntity\Payload::GETTING_WAITING) {
            $this->dev5->debug("|→ FAIL", "NOT-WAITING");
            http_response_code(409);
            exit;
        }

        set_time_limit(300);
        ignore_user_abort(true);
        session_status() === PHP_SESSION_ACTIVE && session_write_close();
        register_shutdown_function(static function () use ($payload) {
            if ($payload->type !== EIEntity\Payload::currentPayload()->type) {
                return;
            }

            $payload->status === EIEntity\Payload::GETTING_RUNNING && $payload->status = EIEntity\Payload::GETTING_WAITING; // phpcs:ignore
            $payload->save();
        });

        $payload->status = EIEntity\Payload::GETTING_RUNNING;
        $payload->save();

        switch ($payload->type) {
            case 'export_categories':
                $finished = CategoryEntity::getInstance()->exportCategories($payload, "|");
                break;
            case 'export_products':
                $finished = ProductEntity::getInstance()->exportProducts($payload, "|");
                break;
            case 'export_orders':
                $finished = OrderEntity::getInstance()->exportOrders($payload, "|");
                break;
            case 'import_categories':
                $finished = CategoryEntity::getInstance()->importCategories($payload, "|");
                break;
            case 'import_products':
                $finished = ProductEntity::getInstance()->importProducts($payload, "|");
                break;
            case 'import_orders':
                $finished = OrderEntity::getInstance()->importOrders($payload, "|");
                break;
            default:
                $this->dev5->debug("|→ FAIL", "type=$payload->type");
                http_response_code(400);
                exit;
        }

        if (EIEntity\Payload::currentPayload()->status !== EIEntity\Payload::GETTING_RUNNING) {
            $this->dev5->debug("|→ FAIL", "NOT-RUNNING");
            http_response_code(409);
            exit;
        }

        if ($finished) {
            $payload->status = EIEntity\Payload::GETTING_FINISHED;
            $this->dev5->debug("|→ SUCCESS", "total=$payload->total");
        } else {
            $payload->status = EIEntity\Payload::GETTING_WAITING;
            $payload->cursor++;
            $this->dev5->debug('|→ SUCCESS', 'NEXT');
        }
    }

    /**
     * @param string $type
     * @return void
     */
    private function eiApproveForm($type)
    {
        $this->document->setTitle($this->dev5->language->get("heading_title_ei_$type"));
        $this->response->setOutput($this->dev5->view([
            'event' => 'approve-form',
            'items' => EIEntity::findMany(null, ['raw' => true]),
            'type' => $type,
            'href_ei_event' => $this->dev5->link('eiEvent', ['type' => $type]),
            'header' => $this->load->controller('common/header'),
            'footer' => $this->load->controller('common/footer')
        ], $this->dev5->route() . "_ei_form"));
    }

    /**
     * @param string $type
     * @return void
     */
    private function eiApproveSubmit($type)
    {
        $payload = EIEntity\Payload::currentPayload();

        if ($payload->type !== $type || $payload->status !== EIEntity\Payload::GETTING_FINISHED) {
            http_response_code(409);
            exit;
        }

        EIEntity::update(['status' => EIEntity::NONE], null);
        ($ids = @$this->request->post['ei']) && EIEntity::update(['status' => EIEntity::PENDING], ['id' => $ids]);

        $payload->status = EIEntity\Payload::APPROVE_WAITING;
        $payload->cursor = 0;
        $payload->total = 0;
        $payload->save();

        $this->response->redirect($this->dev5->link('eiEvent', ['type' => $type, 'event' => 'approve-next-form']));
        exit;
    }

    /**
     * @param EIEntity\Payload $payload
     * @return void
     */
    private function eiApproveNext($payload)
    {
        $this->dev5->debug("• EI-APPROVE-NEXT", "type=$payload->type", "cursor=$payload->cursor");

        if ($payload->status !== EIEntity\Payload::APPROVE_WAITING) {
            http_response_code(409);
            $this->dev5->debug("|→ FAIL", "NOT-WATTING");
            exit;
        }

        set_time_limit(300);
        ignore_user_abort(true);
        session_status() === PHP_SESSION_ACTIVE && session_write_close();
        register_shutdown_function(static function () use ($payload) {
            if ($payload->type !== EIEntity\Payload::currentPayload()->type) {
                return;
            }

            $payload->status === EIEntity\Payload::APPROVE_RUNNING && $payload->status = EIEntity\Payload::APPROVE_WAITING; // phpcs:ignore
            $payload->save();
        });

        $payload->status = EIEntity\Payload::APPROVE_RUNNING;
        $payload->save();

        switch ($payload->type) {
            case 'export_categories':
                $finished = CategoryEntity::getInstance()->exportCategoriesApprove($payload, "|");
                break;
            case 'export_products':
                $finished = ProductEntity::getInstance()->exportProductsApprove($payload, "|");
                break;
            case 'export_orders':
                $finished = OrderEntity::getInstance()->exportOrdersApprove($payload, "|");
                break;
            case 'import_categories':
                $finished = CategoryEntity::getInstance()->importCategoriesApprove($payload, "|");
                break;
            case 'import_products':
                $finished = ProductEntity::getInstance()->importProductsApprove($payload, "|");
                break;
            case 'import_orders':
                $finished = OrderEntity::getInstance()->importOrdersApprove($payload);
                break;
            default:
                http_response_code(400);
                $this->dev5->debug("|→ FAIL", "type=$payload->type");
                exit;
        }

        if (EIEntity\Payload::currentPayload()->status !== EIEntity\Payload::APPROVE_RUNNING) {
            http_response_code(409);
            $this->dev5->debug("|→ FAIL", "NOT-RUNNING");
            exit;
        }

        if ($finished) {
            $payload->status = EIEntity\Payload::APPROVE_FINISHED;
            $this->dev5->debug("|→ SUCCESS", "total=$payload->total");
        } else {
            $payload->status = EIEntity\Payload::APPROVE_WAITING;
            $payload->cursor++;
            $this->dev5->debug('|→ SUCCESS', 'NEXT');
        }
    }

    /**
     * @param string $type
     * @param string $event
     * @return void
     */
    private function eiNextForm($type, $event)
    {
        $next_event = explode('-', $event)[0] . '-next';

        $this->document->setTitle($this->dev5->language->get("heading_title_ei_$type"));
        $this->response->setOutput($this->dev5->view([
            'event' => 'next-form',
            'type' => $type,
            'href_ei_event' => $this->dev5->link('eiEvent', ['type' => $type]),
            'href_ei_event_next' => $this->dev5->link('eiEvent', ['type' => $type, 'event' => $next_event]),
            'header' => $this->load->controller('common/header'),
            'footer' => $this->load->controller('common/footer')
        ], $this->dev5->route() . "_ei_form"));
    }

    /**
     * @param string $type
     * @return void
     */
    private function eiSummary($type)
    {
        $this->document->setTitle($this->dev5->language->get("heading_title_ei_$type"));
        $this->response->setOutput($this->dev5->view([
            'event' => 'summary',
            'type' => $type,
            'header' => $this->load->controller('common/header'),
            'footer' => $this->load->controller('common/footer'),
            'items' => $items = array_map(static function ($item) {
                return $item;
            }, EIEntity::findMany(C::neq('status', EIEntity::NONE), ['raw' => true])),
            'totals' => [
                'success' => count(array_filter($items, static function ($item) {
                    return $item['status'] == EIEntity::SUCCESS;
                })),
                'fail' => count(array_filter($items, static function ($item) {
                    return $item['status'] == EIEntity::FAIL;
                })),
                'queue' => count(array_filter($items, static function ($item) {
                    return $item['status'] == EIEntity::QUEUE;
                })),
                'unknown' => count(array_filter($items, static function ($item) {
                    return $item['status'] != EIEntity::SUCCESS
                        && $item['status'] != EIEntity::FAIL
                        && $item['status'] != EIEntity::QUEUE;
                })),
            ]
        ], $this->dev5->route() . "_ei_form"));
    }
}

\Dev5\OpenCart::compile();
