<?php

/**
 * This file is part of Dev5 Framework for OpenCart
 *
 * @author    Dev5™ <developer@dev5.com.br>
 * @copyright Dev5™ 2025. Todos os direitos reservados.
 * @license   Distribuição/reprodução não autorizada estará sujeita às penas das leis 9609/98 e 9610/98
 * @link      https://dev5.com.br
 * @phpcs:disable
 */

namespace Dev5;

interface IOpenCart
{
}

namespace Dev5\Opencart;

trait Bootstrap
{
    private static $INCOMPATIBLE = [];

    private static function checkPhpVersion(&$i = null)
    {
        return version_compare($i = PHP_VERSION, '5.6', '>=');
    }

    private static function checkExtCurl()
    {
        return extension_loaded('curl');
    }

    private static function checkExtIoncube(&$i = null)
    {
        static $ioncube = null;

        if ($ioncube === null) {
            if (!extension_loaded('ionCube Loader')) {
                $__oc = strtolower(substr(php_uname(), 0, 3));
                $__ln = "ioncube_loader_{$__oc}_" . substr(phpversion(), 0, 3) . (($__oc == 'win') ? '.dll' : '.so');
                if (function_exists('\\dl')) {
                    @\dl($__ln);
                }
                if (!function_exists('\\_il_exec')) {
                    $__ln = "/ioncube/$__ln";
                    $__oid = $__id = realpath(ini_get('extension_dir'));
                    $__here = dirname(__FILE__);
                    if (strlen($__id) > 1 && $__id[1] == ':') {
                        $__id = str_replace('\\', '/', substr($__id, 2));
                        $__here = str_replace('\\', '/', substr($__here, 2));
                    }
                    $__rd = str_repeat('/..', substr_count($__id, '/')) . "$__here/";
                    $__i = strlen($__rd);
                    while ($__i--) {
                        if ($__rd[$__i] == '/') {
                            $__lp = substr($__rd, 0, $__i) . $__ln;
                            if (file_exists("$__oid$__lp")) {
                                $__ln = $__lp;
                                break;
                            }
                        }
                    }
                    if (function_exists('\\dl')) {
                        @\dl($__ln);
                    }
                }
                if (function_exists('_il_exec')) {
                    @\_il_exec();
                }
            }

            $ioncube = extension_loaded('ionCube Loader') &&
                function_exists('\ioncube_loader_version') &&
                version_compare(\ioncube_loader_version(), '10.2', '>=');
        }

        $i = $ioncube ? \ioncube_loader_version() : null;
        return $ioncube;
    }

    private static function checkExtIoncubeXDebug()
    {
        return !(
            version_compare(PHP_VERSION, '8.1', '>=')
            && function_exists('\ioncube_loader_version')
            && version_compare(\ioncube_loader_version(), '13.0', '>=')
            && extension_loaded('xdebug')
        );
    }

    private static function checkExtOpenSSL()
    {
        return extension_loaded('openssl');
    }

    private static function checkMemoryLimit(&$i = null, $again = false)
    {
        static $memory_limit = null;

        $min = 32 * 1024 * 1024; /* 32M */

        if ($memory_limit === null) {
            $memory_limit = @ini_get('memory_limit');

            if (!is_numeric($memory_limit)) {
                $memory_limit = preg_replace_callback('/(-?\d+)(.?)/', static function ($matches) {
                    return $matches[1] * pow(1024, stripos('BKMGT', $matches[2]));
                }, strtoupper($memory_limit));
            }

            $memory_limit = (int)$memory_limit;

            if (!$again && $memory_limit < $min) {
                $memory_limit = null;
                @ini_set('memory_limit', $min);
                return static::checkMemoryLimit($i, true);
            }
        }

        $pow = $memory_limit > 0 ? floor(log($memory_limit, 1024)) : 0;
        $i = number_format($memory_limit / pow(1024, $pow), 2) . ' ' . ['B', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'][$pow];

        return $memory_limit >= $min;
    }

    private static function checkMaxExecutionTime(&$i = null, $again = false)
    {
        static $max_execution_time = null;

        if ($max_execution_time === null) {
            $max_execution_time = (int)@ini_get('max_execution_time') ?: 0;

            if (!$again && $max_execution_time > 0 && $max_execution_time < 10) { /* 10s */
                $max_execution_time = null;
                @ini_set('max_execution_time', '10');
                return static::checkMaxExecutionTime($i, true);
            }
        }

        return ($i = $max_execution_time) < 1 || $i >= 10; /* 10s */
    }

    private static function fnc()
    {
        if ($_GET['FNC'] === 'phpinfo') {
            phpinfo();
            exit;
        }

        if ($_GET['FNC'] === 'trace') {
            $url = "//$_SERVER[HTTP_HOST]$_SERVER[REQUEST_URI]";
            $url = preg_replace(['/&mode=\d+/', '/&clear/', '/&trace=.+/'], '', $url);
            strpos($url, '?') === false && $url .= '?';

            if (array_key_exists('clear', $_GET)) {
                foreach (glob(static::path('/tmp/trace_*.log')) as $i) {
                    @unlink($i);
                }

                header("Location: $url", true, 303);
                exit;
            }

            $mode = intval(static::config('DEBUG'));
            if (array_key_exists('mode', $_GET)) {
                $mode = intval($_GET['mode']);
                if (!in_array($mode, [0, 1, 2])) {
                    $mode = 0;
                }

                static::config(['DEBUG' => $mode]);
                header("Location: $url", true, 303);
                exit;
            }

            echo '<html lang="pt-BR">';
            echo '<title>DEV5 DEBUG</title>';
            echo '<style>*{font-weight:normal;background:#121212;font-family:monospace;color:#fff;text-decoration:none}a{color:#7f7fff}.active{color:red;pointer-events:none}td{padding:5px}tr+tr td{border-top:1px solid}::-webkit-scrollbar {width:4px;height:4px}::-webkit-scrollbar-track{background:#222}::-webkit-scrollbar-thumb{background:#666;border-radius:5px}::-webkit-scrollbar-thumb:hover{background:#444}</style>';
            echo '<h2>';
            echo 'DEV5 DEBUG';

            if (array_key_exists('trace', $_GET)) {
                $path = static::path('/tmp/trace_' . base64_decode($_GET['trace']) . '.log');

                if (stripos($path, static::path('/tmp/trace_')) !== 0) {
                    http_response_code(403);
                    $trace = 'TRACE FORBIDDEN!';
                } elseif (!is_file($path)) {
                    http_response_code(404);
                    $trace = "TRACE \"$path\" NOT FOUND!";
                } else {
                    $trace = file_get_contents($path);
                }

                echo ': trace_' . base64_decode($_GET['trace']) . '.log' . PHP_EOL;
                echo '<small style="float:right;font-size:70%">';
                echo "<a href=\"$url\">← BACK</a>";
                echo '</small>';
                echo "</h2>";
                echo '<pre style="padding:5px;text-wrap:nowrap;overflow-y:auto;height:calc(100vh - 60px)">';
                echo PHP_EOL;
                echo htmlentities($trace);
                echo "</pre>";
                exit;
            }

            echo '<small style="float:right;font-size:70%">';
            echo '<b>MODE</b>: ';
            echo "<a href=\"$url&mode=0\"" . (!$mode ? ' class="active"' : '') . ">DISABLE</a>";
            echo "&ensp;<a href=\"$url&mode=1\"" . ($mode === 1 ? ' class="active"' : '') . ">ENABLE</a>";
            echo "&ensp;<a href=\"$url&mode=2\"" . ($mode === 2 ? ' class="active"' : '') . ">PERSISTS</a>";
            echo '&ensp;/&ensp;';
            echo "<a href=\"$url&clear\">CLEAR</a>";
            echo '</small>';
            echo '</h2>';
            echo '<table>';
            echo '<tbody>';

            $traces = [];
            foreach (glob(static::path('/tmp/trace_*.log')) as $i) {
                $name = substr(basename($i, '.log'), 6);
                $traces[$name] = filemtime($i);
            }
            arsort($traces);
            if (!$traces) {
                echo '<tr><td colspan="2"><i>NO TRACE</i></td></tr>';
            } else {
                foreach ($traces as $name => $i) {
                    echo '<tr>';
                    echo '<td>' . static::datez('Y-m-d H:i:s', date('Y-m-d H:i:s', $i)) . '</td>';
                    echo "<td><a href=\"$url&trace=" . base64_encode($name) . "\">$name</a></td>";
                    echo '</tr>';
                }
            }
            echo '</tbody>';
            echo '</table>';
            echo '</html>';
            exit;
        }
    }

    final public static function bootstrap()
    {
        static $initialized = false;

        if ($initialized) {
            return;
        }

        $initialized = true;

        defined("Dev5\\Opencart\\ROOT") || define("Dev5\\Opencart\\ROOT", __DIR__);

        array_key_exists('FNC', $_GET) && static::fnc();

        static::trace('[' . static::datez() . ']');
        static::trace('• PHP: ' . (static::checkPhpVersion($i) ? 'OK' : 'INVALID') . ", $i");
        static::trace('• ext-curl: ' . (static::checkExtCurl() ? 'OK' : 'INVALID'));
        static::trace('• ext-ioncube: ' . (static::checkExtIoncube($i) ? 'OK' : 'INVALID') . ($i ? ", $i" : ''));
        static::trace('• ext-openssl: ' . (static::checkExtOpenSSL() ? 'OK' : 'INVALID'));
        static::trace('• ini-memory_limit: ' . (static::checkMemoryLimit($i) ? 'OK' : 'INVALID') . ", $i");
        static::trace('• ini-max_execution_time: ' . (static::checkMaxExecutionTime($i) ? 'OK' : 'INVALID') . ", {$i}s");
        static::trace('');

        if (!static::checkPhpVersion()) {
            static::$INCOMPATIBLE[] = 'Necessário que a versão do PHP seja <b>5.6</b> ou superior';
        }

        if (!static::checkExtCurl()) {
            static::$INCOMPATIBLE[] = 'Necessário habilitar a extensão do PHP <b>cUrl</b>';
        }

        if (!static::checkExtIoncube()) {
            static::$INCOMPATIBLE[] = 'Necessário habilitar a extensão do PHP <b>IonCube Loader</b> versão 10.2 ou superior';
        }

        if (!static::checkExtIoncubeXDebug()) {
            static::$INCOMPATIBLE[] = 'Necessário desabilitar a extensão do PHP <b>XDebug</b>, há uma incompatibilidade com a extensão do PHP <b>IonCube Loader v' . (function_exists('\ioncube_loader_version') ? \ioncube_loader_version() : '13.0') . '</b> para <b>PHP' . PHP_MAJOR_VERSION . '</b>';
        }

        if (!static::checkExtOpenSSL()) {
            static::$INCOMPATIBLE[] = 'Necessário habilitar a extensão do PHP <b>OpenSSL</b>';
        }

        if (!static::checkMemoryLimit()) {
            static::$INCOMPATIBLE[] = 'Necessário que a configuração do PHP "memory_limit" seja <b>32M</b> ou superior';
        }

        if (!static::checkMaxExecutionTime()) {
            static::$INCOMPATIBLE[] = 'Necessário que a configuração do PHP "max_execution_time" seja <b>10</b> ou superior';
        }

        if (!static::$INCOMPATIBLE && file_exists(static::path('/sys/gbl.sys'))) {
            try {
                require_once static::path('/sys/gbl.sys');
            } catch (\Exception $e) {
                static::$INCOMPATIBLE['e'] = "<div>UNK_E4632<div style=\"display:none\">$e</div></div>";
            } catch (\Throwable $e) {
                static::$INCOMPATIBLE['e'] = "<div>UNK_E4632<div style=\"display:none\">$e</div></div>";
            } finally {
                unset($e, $GLOBALS['dev5_salt']);
            }
        }

        if (!class_exists("\\Dev5\\OpenCart", false)) {
            static::trace('• EVENT:INITIALIZE-INCOMPATIBLE');

            if (ocver('4', '>=')) {
                static::trace('| • SET:LANGUAGE-OC4X');
                static::trace('| |→ OK');
                static::registry('language')->addPath('extension/dev5', constant('DIR_EXTENSION') . 'dev5/' . ocapp() . '/language/');
            }

            class_alias(OpenCart::class, "\\Dev5\\OpenCart", false);
        }
    }
}

trait Core
{
    final protected static function path($path)
    {
        return preg_replace("/[\/\\\]+/", "/", ROOT . "/$path");
    }

    private static function datez($format = 'Y-m-d H:i:s', $timestamp = 'now')
    {
        $temp = date_create($timestamp, timezone_open(date_default_timezone_get()));

        if (date_default_timezone_get() !== 'America/Sao_Paulo') {
            $temp->setTimezone(timezone_open('America/Sao_Paulo'));
        }

        return $temp->format($format);
    }

    private static function config($_ = null)
    {
        static $local = null;

        $filename = static::path('/config.json');

        if ($local === null && is_file($filename)) {
            $local = @json_decode(file_get_contents($filename), true) ?: null;
        }

        $catalog = null;
        if ($local === null || (ocapp('ADMIN') && !empty($local['PATH']['AGAIN']))) {
            is_file($filename) && @unlink($filename);
            $catalog = !empty($local['PATH']['CATALOG']) ? $local['PATH']['CATALOG'] : null;
            $local = null;
        }

        if ($local === null) {
            $local = [
                'DEBUG' => '0',
                'PATH' => []
            ];

            if (ocapp('ADMIN')) {
                $local['PATH']['ADMIN'] = @basename(constant('DIR_APPLICATION')) ?: 'admin';
                $local['PATH']['CATALOG'] = $catalog ?: @basename(constant('DIR_CATALOG')) ?: 'catalog';
            } else {
                $local['PATH']['ADMIN'] = 'admin';
                $local['PATH']['CATALOG'] = @basename(constant('DIR_APPLICATION')) ?: 'catalog';
                $local['PATH']['AGAIN'] = true;
            }

            @file_put_contents($filename, json_encode($local, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
        }

        if (func_num_args() === 1) {
            if (!is_array($_)) {
                return $local[$_];
            }

            $local = array_merge($local, $_);
            @file_put_contents($filename, json_encode($local, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
        }

        return $local;
    }

    private static function extension_info($extension_id)
    {
        static $info = [];

        if (!isset($info[$extension_id])) {
            $i = static::path("dat/$extension_id");
            $i && $i = is_file($i) ? $i : false;
            $i && $i = @file_get_contents($i) ?: false;
            $i && $i = @preg_replace('/^--.*$/m', '', $i) ?: false;
            $i && $i = @base64_decode($i) ?: false;
            $i && $i = @gzinflate($i) ?: false;
            $i && $i = @json_decode($i, true) ?: false;
            if (is_array($i)) unset($i['_salt']);
            $info[$extension_id] = $i ?: [];
            !isset($info[$extension_id]['name']) && $info[$extension_id]['name'] = $extension_id;
            !isset($info[$extension_id]['type']) && $info[$extension_id]['type'] = 'module';
        }

        return isset($info[$extension_id]) ? $info[$extension_id] : ['name' => $extension_id, 'type' => 'module'];
    }

    final public static function trace()
    {
        static $trace = null;

        if ($trace === false) {
            return;
        }

        if ($trace === null) {
            if (intval(static::config('DEBUG')) === 1) {
                $trace = static::path('/tmp/trace_' . strtolower(ocapp()) . '.log');
                is_file($trace) && @unlink($trace);
            } elseif (intval(static::config('DEBUG')) === 2) {
                $trace = static::path('/tmp/trace_' . strtolower(ocapp()) . '_' . static::datez("Y-m-d-H-i-s") . '_' . uniqid() . '.log');
            } else {
                $trace = false;
                return;
            }

            if (!is_dir(dirname($trace))) {
                @mkdir(dirname($trace), 0755, true);
            }
        }

        foreach (func_get_args() as $i) {
            @error_log(print_r($i, true) . PHP_EOL, 3, $trace);
        }
    }

    final public static function app($compare = null)
    {
        static $app = null;

        if ($app === null) {
            $trace = debug_backtrace();
            $trace = $trace[count($trace) - 1];
            if (!empty($trace['function']) && $trace['function'] === 'start' && !empty($trace['args'][0])) {
                $app = strtoupper($trace['args'][0]) ?: null;
            }
            unset($trace);

            if ($app === null && !empty($GLOBALS['application_config'])) {
                $app = strtoupper($GLOBALS['application_config']) ?: null;
            }

            if ($app === null) {
                $app = defined('DIR_CATALOG') ? 'ADMIN' : 'CATALOG';
            }
        }

        if ($compare !== null) {
            return $app === $compare;
        }

        return $app;
    }

    final public static function version($compare, $operator)
    {
        return version_compare(constant('VERSION'), $compare, $operator);
    }

    final public static function registry($property = null, $value = null)
    {
        static $registry = null;

        if ($registry === null) {
            static::trace('| • GET:REGISTRY');
            if (isset($GLOBALS['registry'])) {
                $registry = $GLOBALS['registry'];
            }

            if (empty($registry)) {
                foreach (debug_backtrace() as $trace) {
                    if (!empty($trace['args']) && is_array($trace['args'])) {
                        foreach ($trace['args'] as $arg) {
                            if (is_object($arg) && in_array('registry', explode('\\', strtolower(get_class($arg))))) {
                                $registry = $arg;
                                break 2;
                            }
                        }
                    }
                }
            }

            static::trace('| |→ ' . ($registry ? 'OK' : 'E2464'));
        }

        if ($registry === null) {
            return null;
        }

        if (func_num_args() < 1) {
            return $registry;
        }

        if (func_num_args() < 2) {
            return $registry->get($property);
        }

        $registry->set($property, $value);
        return true;
    }
}

trait Action
{
    private static function fst($extension_id, $status)
    {
        if (!$extension_id) {
            return;
        }

        $file = static::path("/dat/$extension_id.fst");
        if ($status) {
            $fst = static::path($file);
            $fst && $fst = is_file($fst) ? $fst : false;
            $fst && $fst = @file_get_contents($fst) ?: false;
            $fst && $fst = @base64_decode($fst) ?: false;
            $fst && $fst = @json_decode($fst, true) ?: false;
            if ($fst) {
                array_map(static function ($i) {
                    static::registry('db')->query("UPDATE `" . DB_PREFIX . "setting` SET `value`=1 WHERE `setting_id`='$i[0]' AND `key`='$i[1]'");
                    static::registry('config')->set($i[1], 1);
                }, $fst[0] ?: []);

                array_map(static function ($i) {
                    static::registry('db')->query("UPDATE `" . DB_PREFIX . "setting` SET `value`=REPLACE(`value`,'status\":\"1\"','status\":\"0\"') WHERE `setting_id`='$i[0]' AND `key`='$i[1]'");
                    $j = static::registry('config')->get($i[1]);
                    foreach ($j as $k => $v) {
                        if (stripos($k, 'status') !== false) {
                            $j[$k] = 0;
                        }
                    }
                    static::registry('config')->set($i[1], $j);
                }, $fst[1] ?: []);

                array_map(static function ($i) {
                    static::registry('db')->query("UPDATE `" . DB_PREFIX . "module` SET `setting`=1 WHERE `module_id`='$i[0]' AND `code`='$i[1]'");
                }, $fst[2] ?: []);

                array_map(static function ($i) {
                    static::registry('db')->query("UPDATE `" . DB_PREFIX . "module` SET `setting`=REPLACE(`setting`,'status\":\"0\"','status\":\"1\"') WHERE `module_id`='$i[0]' AND `code`='$i[1]'");
                }, $fst[3] ?: []);

                @unlink($file);
            }
        } else {
            if (file_exists($file)) {
                return;
            }

            $fst = [];
            $fst[0] = array_map(static function ($i) {
                return [$i['setting_id'], $i['key']];
            }, static::registry('db')->query("SELECT `setting_id`,`key` FROM `" . DB_PREFIX . "setting` WHERE `key` LIKE '%d5$extension_id%status' AND `value`=1")->rows);
            $fst[0] && static::registry('db')->query("UPDATE `" . DB_PREFIX . "setting` SET `value`=0 WHERE `key` LIKE '%d5$extension_id%status' AND `value`=1");

            $fst[1] = array_map(static function ($i) {
                return [$i['setting_id'], $i['key']];
            }, static::registry('db')->query("SELECT `setting_id`,`key` FROM `" . DB_PREFIX . "setting` WHERE `key` LIKE '%d5$extension_id%' AND `value` LIKE '%status\":\"1\"%'")->rows);
            $fst[1] && static::registry('db')->query("UPDATE `" . DB_PREFIX . "setting` SET `value`=REPLACE(`value`,'status\":\"1\"','status\":\"0\"') WHERE `key` LIKE '%d5$extension_id%' AND `value` LIKE '%status\":\"1\"%'");

            $fst[2] = array_map(static function ($i) {
                return [$i['module_id'], $i['code']];
            }, static::registry('db')->query("SELECT `module_id`,`code` FROM `" . DB_PREFIX . "module` WHERE `code` LIKE '%d5$extension_id%status' AND `setting`=1")->rows);
            $fst[2] && static::registry('db')->query("UPDATE `" . DB_PREFIX . "module` SET `setting`=0 WHERE `code` LIKE '%d5$extension_id%status' AND `setting`=1");

            $fst[3] = array_map(static function ($i) {
                return [$i['module_id'], $i['code']];
            }, static::registry('db')->query("SELECT `module_id`,`code` FROM `" . DB_PREFIX . "module` WHERE `code` LIKE '%d5$extension_id%' AND `setting` LIKE '%status\":\"1\"%'")->rows);
            $fst[3] && static::registry('db')->query("UPDATE `" . DB_PREFIX . "module` SET `setting`=REPLACE(`setting`,'status\":\"1\"','status\":\"0\"') WHERE `code` LIKE '%d5$extension_id%' AND `setting` LIKE '%status\":\"1\"%'");

            is_dir(dirname($file)) || @mkdir(dirname($file), 0755, true);
            @file_put_contents($file, json_encode($fst, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE));
        }
    }

    private static function action($trace)
    {
        if (
            isset($trace['function'], $trace['class'], $trace['object'])
            && strtolower($trace['function']) === 'execute'
            && strtolower($trace['class']) === 'opencart\system\engine\action'
            && ($json = json_encode((array)$trace['object']))
            && preg_match("~[/\\\]d5~i", $json)
        ) {
            $action = array_merge(['_match' => 8], json_decode(str_ireplace(['\u0000', 'opencart\\\\system\\\\engine\\\\'], '', $json), 1));
            if (!empty($action['Actionroute']) && stripos($action['Actionroute'], '|') !== false) {
                $action['_match'] = 10;
                $action['Actionroute'] = explode('|', $action['Actionroute'], 2)[0];
            } elseif (!empty($action['Actionroute']) && stripos($action['Actionroute'], '.') !== false) {
                $action['_match'] = 11;
                $action['Actionroute'] = explode('.', $action['Actionroute'], 2)[0];
            }
        } elseif (
            isset($trace['function'], $trace['class'], $trace['args'][0])
            && strtolower($trace['function']) === 'controller'
            && strtolower($trace['class']) === 'opencart\system\engine\loader'
            && preg_match("~[/\\\]d5~i", $trace['args'][0])
        ) {
            $trace['args'][0] = stripos($trace['args'][0], '|') !== false
                ? explode('|', $trace['args'][0], 2)
                : explode('.', $trace['args'][0], 2);
            $action = [
                '_match' => 12,
                'Actionroute' => $trace['args'][0][0],
                'Actionmethod' => !empty($trace['args'][0][1]) ? $trace['args'][0][1] : 'index',
                'Actionargs' => [],
                'Actionclass' => null
            ];
        } elseif (
            isset($trace['function'], $trace['class'], $trace['args'][0])
            && strtolower($trace['function']) === 'model'
            && strtolower($trace['class']) === 'opencart\system\engine\loader'
            && preg_match("~[/\\\]d5~i", $trace['args'][0])
        ) {
            $action = [
                '_match' => 9,
                'Actionroute' => $trace['args'][0],
                'Actionmethod' => 'model',
                'Actionargs' => [],
                'Actionclass' => 'model' . preg_replace('/[^a-zA-Z\d]/', '', $trace['args'][0])
            ];
        } elseif (
            isset($trace['function'], $trace['class'], $trace['object'])
            && strtolower($trace['function']) === 'execute'
            && strtolower($trace['class']) === 'action'
            && ($json = json_encode((array)($trace['object'])))
            && preg_match("~[/\\\]d5~i", $json)
        ) {
            $action = array_merge(['_match' => 1], json_decode(str_ireplace('\u0000', '', $json), 1));
            if (empty($action['Actionroute'])) {
                if (isset($action['Actionid'])) {
                    $action['_match'] = 4;
                    $action['Actionroute'] = $action['Actionid'];
                    $action['Actionmethod'] = 'index';
                    if (stripos($action['Actionroute'], '/d5') !== strripos($action['Actionroute'], '/')) {
                        $action['_match'] = 5;
                        $action['Actionmethod'] = substr($action['Actionroute'], strripos($action['Actionroute'], '/') + 1);
                        $action['Actionroute'] = substr($action['Actionroute'], 0, strripos($action['Actionroute'], '/'));
                    }
                } elseif (isset($action['file'])) {
                    if (isset($action['method'])) {
                        $action['Actionmethod'] = $action['method'];
                        unset($action['method']);
                    }
                    $action['_match'] = 7;
                    $action['Actionfile'] = realpath($action['file']);
                    $action['Actionroute'] = str_ireplace([str_ireplace('\\', '/', realpath(constant('DIR_APPLICATION'))) . '/controller/', '.php'], '', str_ireplace('\\', '/', realpath($action['Actionfile'])));
                } else {
                    $action['_match'] = 6;
                    $action['Actionroute'] = str_ireplace([str_ireplace('\\', '/', realpath(constant('DIR_APPLICATION'))) . '/controller/', '.php'], '', str_ireplace('\\', '/', realpath($action['Actionfile'])));
                }
            }
        } elseif (
            isset($trace['function'], $trace['class'], $trace['args'][0])
            && strtolower($trace['function']) === 'model'
            && strtolower($trace['class']) === 'loader'
            && preg_match("~[/\\\]d5~i", $trace['args'][0])
        ) {
            $action = [
                '_match' => 3,
                'Actionroute' => $trace['args'][0],
                'Actionmethod' => 'model',
                'Actionargs' => [],
                'Actionclass' => 'model' . preg_replace('/[^a-zA-Z0-9]/', '', $trace['args'][0])
            ];
        } elseif (
            isset($trace['function'], $trace['class'], $trace['args'][0])
            && strtolower($trace['function']) === 'controller'
            && strtolower($trace['class']) === 'loader'
            && preg_match("~[/\\\]d5~i", $trace['args'][0])
        ) {
            $route = $trace['args'][0];
            $method = 'index';
            $m1 = stripos($route, '/d5');
            $m1 === false && $$m1 = stripos($route, '\\d5');
            $m2 = strripos($route, '/');
            if ($m1 !== $m2) {
                $method = substr($route, $m2 + 1);
                $route = substr($route, 0, $m2);
            }
            $action = [
                '_match' => 2,
                'Actionroute' => $route,
                'Actionmethod' => $method,
                'Actionargs' => isset($trace['args'][1]) ? $trace['args'][1] : []
            ];
        }

        if (!isset($action)) {
            return false;
        }

        $class = $route = $method = $match = null;
        if ($action) {
            $action = @json_decode(str_ireplace('\u0000', '', json_encode($action)), 1);
            $class = !empty($action['Actionclass']) ? $action['Actionclass'] : '';
            $route = !empty($action['Actionroute']) ? $action['Actionroute'] : '';
            $method = !empty($action['Actionmethod']) ? $action['Actionmethod'] : '';
            $match = !empty($action['_match']) ? $action['_match'] : '';
            if (!$route && ocver('2.2', '<')) {
                $route = @str_replace([constant('DIR_APPLICATION'), 'controller/', '.php'], '', $action['Actionfile']);
                if ($route === 'extension/module' && ($method === 'install' || $method === 'uninstall')) {
                    $route = @str_ireplace('extension/', '', $route) . "/$_REQUEST[extension]";
                }
            }

            if (!$method) {
                $method = 'index';
            }
        }

        return [$match, $class, $route, $method, $action];
    }

    private static function action_extension_id()
    {
        foreach (array_reverse(get_required_files()) as $file) {
            foreach (['php', 'tpl', 'twig'] as $ext) {
                $file = basename($file, ".$ext");
            }
            if (substr($file, 0, 2) === 'd5') {
                return substr($file, 2);
            }
        }
        return null;
    }
}

#[\AllowDynamicProperties]
class Silent implements \ArrayAccess, \Iterator, \JsonSerializable, \Dev5\IOpenCart
{
    #[\ReturnTypeWillChange]
    final public function getMethod()
    {
        return [];
    }

    #[\ReturnTypeWillChange]
    final public function getQuote()
    {
        return [];
    }

    #[\ReturnTypeWillChange]
    final public function __get($a)
    {
        return [];
    }

    #[\ReturnTypeWillChange]
    final public function __set($a, $b)
    {
    }

    #[\ReturnTypeWillChange]
    final public function __call($a, $b)
    {
        return [];
    }

    #[\ReturnTypeWillChange]
    final public static function __callStatic($a, $b)
    {
        return [];
    }

    #[\ReturnTypeWillChange]
    final public function offsetExists($offset)
    {
        return false;
    }

    #[\ReturnTypeWillChange]
    final public function offsetGet($offset)
    {
        return [];
    }

    #[\ReturnTypeWillChange]
    final public function offsetSet($offset, $value)
    {
    }

    #[\ReturnTypeWillChange]
    final public function offsetUnset($offset)
    {
    }

    #[\ReturnTypeWillChange]
    public function current()
    {
        return [];
    }

    #[\ReturnTypeWillChange]
    public function next()
    {
    }

    #[\ReturnTypeWillChange]
    public function key()
    {
        return null;
    }

    #[\ReturnTypeWillChange]
    public function valid()
    {
        return false;
    }

    #[\ReturnTypeWillChange]
    public function rewind()
    {
    }

    #[\ReturnTypeWillChange]
    final public function jsonSerialize()
    {
        return [];
    }

    #[\ReturnTypeWillChange]
    final public function __invoke()
    {
        return [];
    }

    #[\ReturnTypeWillChange]
    final public function __toString()
    {
        return '';
    }
}

abstract class OpenCart implements \Dev5\IOpenCart
{
    use Bootstrap;
    use Core;
    use Action;

    final public static function compile(&$_ = null)
    {
        func_num_args() > 0 ? static::language($_) : static::code();
    }

    private static function language(&$_)
    {
        static $language = [];

        $start_time = microtime(true);
        static::trace('• EVENT:LANGUAGE');

        static::trace('| • GET:EXTENSION');
        {
            $extension_id = static::action_extension_id();
            if (!$extension_id) {
                static::trace('| |→ E4630');
                static::trace('|→ ERROR, ' . round(microtime(true) - $start_time, 3) . 's');
                return;
            }
            static::trace('| |→ OK, extension_id=' . $extension_id);
        }

        static::trace('| • GET:IS-CACHED');
        {
            if (isset($language[$extension_id])) {
                $_ = array_merge($_, $language[$extension_id]);
                static::trace('| |→ TRUE');
                static::trace('|→ OK, ' . round(microtime(true) - $start_time, 3) . 's');
                return;
            }
            static::trace('| |→ FALSE');
        }

        static::trace('| • GET:EXTENSION-INFO');
        {
            $info = static::extension_info($extension_id);
            if (!$info) {
                static::trace('| |→ E4631');
                static::trace('|→ ERROR, ' . round(microtime(true) - $start_time, 3) . 's');
                return;
            }
            static::trace('| |→ OK');
        }

        static::trace('| • GET:IS-DESIGN');
        {
            $is_design = false;
            $route = isset($_REQUEST['route']) ? $_REQUEST['route'] : '';
            !$is_design || $is_design = stripos($route, 'design/layout/') !== false;
            !$is_design || $is_design = stripos($route, 'journal3/') !== false;
            !$is_design || $is_design = stripos($route, 'journal2/') !== false;
            !$is_design || $is_design = stripos($route, '/compraexpress') !== false;
            static::trace('| |→ ' . ($is_design ? 'TRUE' : 'FALSE'));
        }

        $sub_title = (isset($info['version']) ? " Versão: $info[version]" : '') . ' por Dev5&trade;';
        if (!$is_design) {
            $update_alert = '';
            if (ocapp('ADMIN') && file_exists(static::path("/dat/$extension_id.upd"))) {
                $update = @json_decode(file_get_contents(static::path("/dat/$extension_id.upd")), 1) ?: [];

                $route = (ocver('2.3', '>=') ? 'extension/' : '');
                $route .= (ocver('4', '>=') ? 'dev5/' : '');
                $route .= "$info[type]/d5$extension_id";
                $token = ($token = (ocver('3', '>=') ? 'user_' : '') . 'token') . '=' . (@static::registry('session')->data[$token] ?: '');
                $args = $token . (ocver('2.3', '>=') ? "&type=$info[type]" : '');

                if (empty($update['read']) && version_compare(@$info['version'] ?: '0', @$update['version'] ?: '0', '<')) {
                    $update_alert = "<a id=\"dev5-$extension_id-update-flag\" href=\"" . static::registry('url')->link($route, $args, ocver('2.2', '<') ? 'SSL' : true) . "#dev5-$extension_id-changelog\" style=\"height:1px;overflow:visible;display:inline-block\">"
                        . "<span style=\"line-height:5px!important;display:inline-block!important;font-size:11px!important;opacity: .8!important;cursor:pointer!important;color:#fff!important;font-weight:400!important;background:#4cb64c!important;padding:3px 5px;position:absolute!important;top:-15px!important;border-radius:5px!important;margin-left:7px!important;left:100%!important\">"
                        . "<i class=\"fa fa-info-circle\"></i> Nova versão disponível"
                        . "</span>"
                        . "</a>";
                }
            }

            $sub_title = "<style><!-- h1 aaa a[href=\"https://dev5.com.br/\"]{position:absolute} --></style>"
                . "<aaa style=\"position:relative!important;display:flex!important;visibility:visible!important;opacity:1!important;white-space:nowrap!important\">"
                . "<a href=\"https://dev5.com.br/\" target=\"_blank\" title=\"Ir para Dev5&trade;\" style=\"line-height:5px!important;display:block!important;font-size:9px!important;opacity:.5!important;cursor:pointer!important;color:#545454!important;font-weight:bold\">"
                . (isset($info['version']) ? "<span style=\"float:left\">Versão: $info[version]&nbsp;</span>" : '') . 'por Dev5&trade;'
                . "</a>"
                . $update_alert
                . "</aaa>";
        }

        $language[$extension_id] = [];
        if (ocapp('ADMIN')) {
            $language[$extension_id]['heading_title'] =
            $language[$extension_id]['heading_title_outer'] = ($is_design ? $info['name'] : "<aaa style=\"display:inline-grid\"><b>$info[name]</b> $sub_title</aaa>");
            $language[$extension_id]['heading_title_inner'] = $info['name'];
            if (!empty($info['parent'])) {
                $uid = "d5child" . uniqid();
                /** @noinspection ALL */
                $language[$extension_id]['heading_title'] =
                    "<script type=\"text/javascript\" id=\"$uid\">"
                    . "let $uid = document.querySelector('#$uid');"
                    . "if ($uid.parentNode.parentNode.nodeName === 'TR') $uid.parentNode.parentNode.parentNode.removeChild($uid.parentNode.parentNode);"
                    . "$uid.parentNode.removeChild($uid);"
                    . "delete $uid;"
                    . "</script>"
                    . $language[$extension_id]['heading_title'];
            }
        }

        $_ = array_merge($_, $language[$extension_id]);
        static::trace('|→ OK, ' . round(microtime(true) - $start_time, 3) . 's');
    }

    private static function code()
    {
        $start_time = microtime(true);
        static::trace('• EVENT:ACTION');

        #region GET:EXTENSION
        static::trace('| • GET:EXTENSION');
        foreach (debug_backtrace() as $i) {
            if ($action = static::action($i)) {
                break;
            }
        }

        list($match, $class, $route, $method) = empty($action) ? [null, null, null, null] : $action;
        if (!$route || !$method) {
            static::trace("| |→ E4635, route=$route, method=$method" . (!empty($class) ? ", class=$class" : ''));
            static::trace('|→ ERROR, ' . round(microtime(true) - $start_time, 3) . 's');
            return;
        }

        $extension_id = explode('.', substr(basename($route), 2))[0];
        static::trace("| |→ OK, match=$match, extension_id=$extension_id, route=$route, method=$method");
        #endregion GET:EXTENSION

        @error_reporting(0);
        @ini_set('error_reporting', 0);
        @ini_set('display_errors', 0);
        @ini_set('log_errors', 0);
        $extension_id && static::fst($extension_id, false);

        if (!static::$INCOMPATIBLE) {
            static::$INCOMPATIBLE = [$GLOBALS['dev5_last_error'] = 'E4636'];
        }

        $extension_info = self::extension_info($extension_id);
        static::trace('| • SET:DISABLE_MODE');

        if ($method === 'uninstall' || ocapp('CATALOG')) {
            static::trace('| |→ SILENT');

            if (isset($extension_info['classes'][strtolower(ocapp())])) {
                array_map(static function ($class) {
                    class_exists($class, false) || class_alias(Silent::class, $class);
                }, $extension_info['classes'][strtolower(ocapp())] ?: []);
            }

            error_log(
                "Atenção, seu servidor não atende aos requisitos mínimos da extensão " . (isset($extension_info['name']) ? $extension_info['name'] : "<$extension_id>") . "\n"
                . "Verifique com o suporte da sua hospedagem ou seu WebMaster os seguintes requisitos para continuar a instalação\n"
                . "• " . implode("\n• ", static::$INCOMPATIBLE) . "\n"
                . "\n"
                . "*A extensão foi desabilitada por segurança e para evitar possíveis erros.\n"
                . "*Assim que os requisitos forem atendidos a extensão poderá ser reativada e continuará configurada."
            );

            static::trace('|→ OK, ' . round(microtime(true) - $start_time, 3) . 's');
        } else {
            static::trace('| |→ ALERT');

            $route = isset($_REQUEST['route']) ? $_REQUEST['route'] : null;
            $installing = ocver('4', '>=')
                ? stripos($route, "extension/$extension_info[type]|install") !== false || stripos($route, "extension/$extension_info[type].install") !== false
                : (ocver('2.3', '<')
                    ? stripos($route, "extension/$extension_info[type]/install") !== false
                    : stripos($route, "extension/extension/$extension_info[type]/install") !== false
                );

            if ($extension_id && $installing) {
                // Prevents the extension from being installed if the server does not meet the minimum requirements
                static::registry('db')->query("DELETE FROM `" . DB_PREFIX . "extension` WHERE `code` LIKE '%d5$extension_id%'");
            }

            if (ocver('4', '>=')) {
                if (stripos($_SERVER['HTTP_ACCEPT'], 'application/json') !== false) {
                    ob_start();
                } else {
                    $installing = false;
                }
            } elseif (ocver('2.3', '>=')) {
                if (empty($_SERVER['HTTP_X_REQUESTED_WITH'])) {
                    $installing = false;
                }
            }

            if (ocver('2.3', '<') || !$installing) {
                static::registry('document')->setTitle('Incompatibilidade - Dev5™');
                echo static::registry('load')->controller('common/header');
                echo "<div id=\"content\"><div class=\"container-fluid\"><br>";
            }

            echo "<div class=\"alert alert-danger\">"
                . "<i class=\"fa fa-exclamation-circle\"></i> <b>Atenção, seu servidor não atende aos requisitos mínimos da extensão $extension_info[name]</b><br>"
                . "<p style=\"margin-bottom:.5rem\">Verifique com o suporte da sua hospedagem ou seu WebMaster os seguintes requisitos para continuar a instalação</p>"
                . "&emsp;• " . implode("<br>&emsp;• ", static::$INCOMPATIBLE);

            if ($extension_id && !$installing) {
                echo "<br><br><br><small><i>*A extensão foi desabilitada por segurança e para evitar possíveis erros.<br>"
                    . "*Assim que os requisitos forem atendidos a extensão poderá ser reativada e continuará configurada.</i></small>";
            }

            echo '</div><div style="float:right">';
            if ($extension_id) {
                $route = ocver('3', '>=') ? 'marketplace/extension' : (ocver('2.3', '>=') ? 'extension/extension' : "extension/$extension_info[type]");
                $token = ($token = (ocver('3', '>=') ? 'user_' : '') . 'token') . '=' . (@static::registry('session')->data[$token] ?: '');
                $args = $token . (ocver('2.3', '>=') ? "&type=$extension_info[type]" : '');
                echo "<a class=\"btn btn-default btn-light\" href=\"" . static::registry('url')->link($route, $args, ocver('2,2', '<') ? 'SSL' : true) . "\">" . static::registry('language')->get('button_continue') . "</a>";
            }
            echo '</div>';

            if (ocver('2.3', '<') || !$installing) {
                echo "</div></div>";
                echo static::registry('load')->controller('common/footer');
            }

            if (ocver('4', '>=') && stripos(@$_SERVER['HTTP_ACCEPT'], 'application/json') !== false) {
                ob_get_clean();
                header('content-type: application/json; charset=utf8');
                echo json_encode([
                    'error' => "<b><small>Atenção, seu servidor não atende aos requisitos mínimos da extensão $extension_info[name]</small></b><br>"
                        . "<p style=\"margin-bottom:.5rem\"><small>Verifique com o suporte da sua hospedagem ou seu WebMaster os seguintes requisitos para continuar a instalação</small></p>"
                        . "&emsp;• " . implode("<br>&emsp;• ", static::$INCOMPATIBLE)
                ]);
            }

            static::trace('|→ OK, ' . round(microtime(true) - $start_time, 3) . 's');
            exit;
        }
    }
}

function ocver($compare, $operator)
{
    return OpenCart::version($compare, $operator);
}

function ocapp($compare = null)
{
    return OpenCart::app($compare);
}

class_exists("\\Dev5\\OpenCart", false) || OpenCart::bootstrap();
